<?php

namespace Intergo\VerifyClient;

use Intergo\BaseClient\Responses\BaseResponseInterface;

abstract class AbstractVerifyApiService
{
    /**
     * List of possible 2FA statuses.
     */
    const PENDING = 'pending';
    const VERIFIED = 'verified';
    const FAILED = 'failed';

    /** @var VerifierClient */
    protected $client;


    /** @var string */
    protected $token;

    public function __construct(string $baseUrl, string $traceId, string $token = null)
    {
        $this->client = new VerifierClient($baseUrl, $traceId);
        $this->token = $token;
    }

    /**
     * @param string $action
     * @param BaseResponseInterface $result
     * @return BaseResponseInterface
     */
    abstract public function response(string $action, BaseResponseInterface $result): BaseResponseInterface;

    /**
     * @param string $endpoint
     * @param array $urlParams
     * @param array $bodyParams
     * @return BaseResponseInterface
     */
    protected function execute(string $endpoint, array $urlParams = [], array $bodyParams = []): BaseResponseInterface
    {
        return $this->response($endpoint, $this->client->execute($endpoint, $urlParams, $bodyParams));
    }

    /**
     * @param $endpoint
     * @param array $urlParams
     * @param array $bodyParams
     * @return BaseResponseInterface
     */
    protected function executeAuth($endpoint, array $urlParams = [], array $bodyParams = []): BaseResponseInterface
    {
        $result = $this->client->executeAuth($endpoint, $this->token, $urlParams, $bodyParams);
        return $this->response($endpoint, $result);
    }

    /***********************Applications****************************/
    public function getApplication($id): BaseResponseInterface
    {
        return $this->execute(__FUNCTION__, compact('id'));
    }

    public function getApplications(): BaseResponseInterface
    {
        return $this->execute(__FUNCTION__);
    }

    public function createSystem(array $userData): BaseResponseInterface
    {
        return $this->executeAuth(__FUNCTION__, [], $userData);
    }

    public function confirmSystem(string $id, string $password): BaseResponseInterface
    {
        return $this->execute(__FUNCTION__, [], compact('id', 'password'));
    }

    public function getGuid(string $recipient): BaseResponseInterface
    {
        return $this->execute(__FUNCTION__, [], compact('recipient'));
    }

    /**
     * Retrieves all the applications of logged-in user
     * @return BaseResponseInterface
     */
    public function getMyApplications($userId): BaseResponseInterface
    {
        return $this->execute(__FUNCTION__, compact('userId'));
    }

    /**
     * @param $name
     * @param $user_id
     * @return BaseResponseInterface
     */
    public function storeApplication($name, $user_id): BaseResponseInterface
    {
        return $this->execute(__FUNCTION__, [], compact('name', 'user_id'));
    }

    /**
     * @param $id
     * @param $name
     * @param $user_id
     * @return BaseResponseInterface
     */
    public function updateApplication($id, $name, $user_id): BaseResponseInterface
    {
        return $this->execute(__FUNCTION__, compact('id'), compact('name', 'user_id'));
    }

    /**
     * @param $application
     * @return BaseResponseInterface
     */
    public function getUserSettingsApplication($application): BaseResponseInterface
    {
        return $this->executeAuth(__FUNCTION__, compact('application'));
    }

    /**
     * @param $application
     * @param $data
     * @return BaseResponseInterface
     */
    public function updateUserSettingsApplication($application, $data): BaseResponseInterface
    {
        return $this->executeAuth(__FUNCTION__, compact('application'), $data);
    }

    public function deleteApplication($id): BaseResponseInterface
    {
        return $this->execute(__FUNCTION__, compact('id'));
    }

    public function deleteApplicationChannelSettings($id, $channel): BaseResponseInterface
    {
        return $this->execute(__FUNCTION__, compact('id', 'channel'));
    }

    /***********************Applications Settings****************************/
    public function storeApplicationSetting($channel, $sender_id, $min_length, $max_length, $max_tries, $ttl, $password_type, $callback_url, $application_id): BaseResponseInterface
    {
        return $this->execute(__FUNCTION__, [], compact(
            'channel',
            'sender_id',
            'min_length',
            'max_length',
            'max_tries',
            'ttl',
            'password_type',
            'callback_url',
            'application_id'));
    }

    /***********************Billing Settings****************************/
    public function storeBillingSetting($channel, $our_cost, $cost_success, $cost_fail, $application_id): BaseResponseInterface
    {
        return $this->execute(__FUNCTION__, [], compact(
            'channel',
            'our_cost',
            'cost_success',
            'cost_fail',
            'application_id'));
    }

    public function storeLocalizeBillingSettings($channel, $application_id, $billing): BaseResponseInterface
    {
        return $this->execute(__FUNCTION__, [], compact(
            'channel',
            'application_id',
            'billing'));
    }

    /***********************Templates****************************/

    public function storeTemplate($locale, $channel, $title, $message, $application_id): BaseResponseInterface
    {
        return $this->execute(__FUNCTION__, [], compact('locale',
            'channel',
            'title',
            'message',
            'application_id'));
    }

    public function storeLocalizeTemplates($channel, $application_id, $templates): BaseResponseInterface
    {
        return $this->execute(
            __FUNCTION__,
            [],
            compact(
                'channel',
                'application_id',
                'templates')
        );
    }
    /***********************Templates****************************/


    /**
     * @return BaseResponseInterface
     */
    public function getRecipient()
    {
        return $this->execute(__FUNCTION__);
    }

    public function list($queryString): BaseResponseInterface
    {
        if (is_string($queryString)) {
            parse_str($queryString, $queryString);
        }
        extract($queryString);
        unset($queryString);
        return $this->execute(__FUNCTION__, [], get_defined_vars());
    }

    public function listUser($userId, $queryString): BaseResponseInterface
    {
        if (is_string($queryString)) {
            parse_str($queryString, $queryString);
        }
        extract($queryString);
        unset($queryString);
        return $this->execute(__FUNCTION__, compact('userId'), get_defined_vars());
    }

    public function statistics($userId, $queryString): BaseResponseInterface
    {
        if (is_string($queryString)) {
            parse_str($queryString, $queryString);
        }
        extract($queryString);
        unset($queryString);
        return $this->execute(__FUNCTION__, compact('userId'), get_defined_vars());
    }

    public function export($userId, $queryString = null): BaseResponseInterface
    {
        if (is_string($queryString)) {
            parse_str($queryString, $queryString);
        }
        extract($queryString);
        unset($queryString);
        return $this->executeAuth(__FUNCTION__, compact('userId'), get_defined_vars());
    }

    public function managerExport($userId, $queryString = null): BaseResponseInterface
    {
        if (is_string($queryString)) {
            parse_str($queryString, $queryString);
        }
        extract($queryString);
        unset($queryString);
        return $this->executeAuth(__FUNCTION__, compact('userId'), get_defined_vars());
    }

    /***********************Failed Jobs****************************/
    public function getFailedJobs($start = null, $end = null, $failed_queue = null, $page = 1)
    {
        return $this->execute(__FUNCTION__, [], compact('start', 'end', 'failed_queue', 'page'));
    }

    public function getFailedJob($id)
    {
        return $this->execute(__FUNCTION__, compact('id'));
    }

    public function getFailedJobsQueues()
    {
        return $this->execute(__FUNCTION__, [], get_defined_vars());
    }

    public function retryFailedJobs($ids)
    {
        return $this->execute(__FUNCTION__, [], get_defined_vars());
    }

    public function deleteFailedJobs($ids)
    {
        return $this->execute(__FUNCTION__, [], get_defined_vars());
    }
}
