<?php

namespace Intergo\VerifyClient;

use Intergo\BaseClient\Responses\BaseResponseInterface;

abstract class VerifierServiceManager
{
    /**
     * List of possible 2FA statuses.
     */
    const PENDING = 'pending';
    const VERIFIED = 'verified';
    const FAILED = 'failed';

    /** @var VerifierClient */
    protected $client;


    /** @var string */
    protected $token;

    public function __construct(string $baseUrl, string $traceId, string $token = null)
    {
        $this->client = new VerifierClient($baseUrl, $traceId);
        $this->token = $token;
    }

    /**
     * @param string $action
     * @param BaseResponseInterface $result
     * @return BaseResponseInterface
     */
    abstract public function response(string $action, BaseResponseInterface $result): BaseResponseInterface;

    /**
     * @param string $endpoint
     * @param array $urlParams
     * @param array $bodyParams
     * @return BaseResponseInterface
     */
    protected function execute(string $endpoint, array $urlParams = [], array $bodyParams = []): BaseResponseInterface
    {
        return $this->response($endpoint, $this->client->execute($endpoint, $urlParams, $bodyParams));
    }

    /**
     * @param $endpoint
     * @param  array  $urlParams
     * @param  array  $bodyParams
     * @return BaseResponseInterface
     */
    protected function executeAuth($endpoint, array $urlParams = [], array $bodyParams = []): BaseResponseInterface
    {
        $result = $this->client->executeAuth($endpoint, $this->token, $urlParams, $bodyParams);
        return $this->response($endpoint, $result);
    }

    /***********************Applications****************************/
    public function getApplication($id): BaseResponseInterface
    {
        return $this->executeAuth(__FUNCTION__, compact('id'));
    }
    public function getApplications(): BaseResponseInterface
    {
        return $this->executeAuth(__FUNCTION__);
    }
    public function createSystem(array $userData): BaseResponseInterface
    {
        return $this->executeAuth(__FUNCTION__, [], $userData);
    }
    public function confirmSystem(string $id, string $password): BaseResponseInterface
    {
        return $this->execute(__FUNCTION__, [], compact('id', 'password'));
    }
    public function getGuid(string $recipient): BaseResponseInterface
    {
        return $this->execute(__FUNCTION__, [], compact('recipient'));
    }

    /**
     * Retrieves all the applications of logged-in user
     * @return BaseResponseInterface
     */
    public function getMyApplications(): BaseResponseInterface
    {
        return $this->executeAuth(__FUNCTION__);
    }

    public function storeApplication($name, $user_id): BaseResponseInterface
    {
        return $this->executeAuth(__FUNCTION__, [],compact('name', 'user_id'));
    }
    public function updateApplication($id, $name, $user_id): BaseResponseInterface
    {
        return $this->executeAuth(__FUNCTION__, compact('id'),compact('name', 'user_id'));
    }
    public function deleteApplication($id): BaseResponseInterface
    {
        return $this->executeAuth(__FUNCTION__, compact('id'));
    }
    /***********************Applications Settings****************************/
    public function getApplicationSetting($id): BaseResponseInterface
    {
        return $this->executeAuth(__FUNCTION__, compact('id'));
    }
    public function storeApplicationSetting($channel, $min_length, $max_length, $max_tries, $ttl, $password_type, $callback_url, $application_id): BaseResponseInterface
    {
        return $this->executeAuth(__FUNCTION__, [], compact(
            'channel',
            'min_length',
            'max_length',
            'max_tries',
            'ttl',
            'password_type',
            'callback_url',
            'application_id'));
    }
    public function updateApplicationSetting($id, $channel, $min_length, $max_length, $max_tries, $ttl, $password_type, $callback_url, $application_id): BaseResponseInterface
    {
        return $this->executeAuth(__FUNCTION__, compact('id'),compact(
            'channel',
            'min_length',
            'max_length',
            'max_tries',
            'ttl',
            'password_type',
            'callback_url',
            'application_id'));
    }
    public function deleteApplicationSetting($id): BaseResponseInterface
    {
        return $this->executeAuth(__FUNCTION__, compact('id'));
    }
    /***********************Billing Settings****************************/
    public function getBillingSetting($id): BaseResponseInterface
    {
        return $this->executeAuth(__FUNCTION__, compact('id'));
    }
    public function storeBillingSetting($channel, $our_cost, $cost_success, $cost_fail, $application_id): BaseResponseInterface
    {
        return $this->executeAuth(__FUNCTION__, [], compact(
            'channel',
            'our_cost',
            'cost_success',
            'cost_fail',
            'application_id'));
    }
    public function storeLocalizeBillingSettings($channel, $application_id, $billing): BaseResponseInterface
    {
        return $this->executeAuth(__FUNCTION__, [], compact(
            'channel',
            'application_id',
            'billing'));
    }
    public function updateBillingSetting($id, $channel, $our_cost, $cost_success, $cost_fail, $application_id): BaseResponseInterface
    {
        return $this->executeAuth(__FUNCTION__, compact('id'),compact(
            'channel',
            'our_cost',
            'cost_success',
            'cost_fail',
            'application_id'));
    }
    public function deleteBillingSetting($id): BaseResponseInterface
    {
        return $this->executeAuth(__FUNCTION__, compact('id'));
    }
    /***********************Templates****************************/
    public function getTemplate($id): BaseResponseInterface
    {
        return $this->executeAuth(__FUNCTION__, compact('id'));
    }
    public function storeTemplate($locale, $channel, $title, $message, $application_id): BaseResponseInterface
    {
        return $this->executeAuth(__FUNCTION__, [],compact('locale',
            'channel',
            'title',
            'message',
            'application_id'));
    }
    public function storeLocalizeTemplates($channel, $application_id, $templates): BaseResponseInterface
    {
        return $this->executeAuth(
            __FUNCTION__,
            [],
            compact(
            'channel',
            'application_id',
            'templates')
        );
    }
    public function updateTemplate($id, $locale, $channel, $title, $message, $application_id): BaseResponseInterface
    {
        return $this->executeAuth(__FUNCTION__, compact('id'),compact('locale',
            'channel',
            'title',
            'message',
            'application_id'));
    }
    public function deleteTemplate($id): BaseResponseInterface
    {
        return $this->executeAuth(__FUNCTION__, compact('id'));
    }
    /***********************Templates****************************/
    public function create($guid, $recipient, $reference, $callback_url, $channel, $locale = null): BaseResponseInterface
    {
        return $this->executeAuth(__FUNCTION__, [], compact(
            'guid',
            'recipient',
            'reference',
            'callback_url',
            'channel',
            'locale',
        ));
    }
    public function confirm($id, $password): BaseResponseInterface
    {
        return $this->executeAuth(__FUNCTION__, [], compact('id', 'password'));
    }
    public function cancel($id): BaseResponseInterface
    {
        return $this->executeAuth(__FUNCTION__, [], compact('id'));
    }
    public function status($id): BaseResponseInterface
    {
        return $this->executeAuth(__FUNCTION__, compact('id'));
    }
    public function list($queryString): BaseResponseInterface
    {
        if (is_string($queryString)) {
            parse_str($queryString, $queryString);
        }
        extract($queryString);
        unset($queryString);
        return $this->executeAuth(__FUNCTION__, [], get_defined_vars());
    }

    /**
     * @return BaseResponseInterface
     */
    public function getRecipient(){
        return $this->executeAuth(__FUNCTION__);
    }

    public function createInternal(
        $guid,
        $recipient,
        $reference,
        $callback_url,
        $channel,
        $locale = null
    ): BaseResponseInterface
    {
        return $this->executeAuth(__FUNCTION__, [], compact(
            'guid',
            'recipient',
            'reference',
            'callback_url',
            'channel',
            'locale',
        ));
    }
    public function confirmInternal($id, $password): BaseResponseInterface
    {
        return $this->executeAuth(__FUNCTION__, [], compact('id', 'password'));
    }
    public function cancelInternal($id): BaseResponseInterface
    {
        return $this->executeAuth(__FUNCTION__, [], compact('id'));
    }
    public function statusInternal($id): BaseResponseInterface
    {
        return $this->executeAuth(__FUNCTION__, compact('id'));
    }
    public function listInternal($queryString): BaseResponseInterface
    {
        if (is_string($queryString)) {
            parse_str($queryString, $queryString);
        }
        extract($queryString);
        unset($queryString);
        return $this->executeAuth(__FUNCTION__, [], get_defined_vars());
    }

    public function listUserInternal($queryString): BaseResponseInterface
    {
        if (is_string($queryString)) {
            parse_str($queryString, $queryString);
        }
        extract($queryString);
        unset($queryString);
        return $this->executeAuth(__FUNCTION__, [], get_defined_vars());
    }

    public function statisticsInternal($queryString): BaseResponseInterface
    {
        if (is_string($queryString)) {
            parse_str($queryString, $queryString);
        }
        extract($queryString);
        unset($queryString);
        return $this->executeAuth(__FUNCTION__, [], get_defined_vars());
    }

    public function export($queryString = null): BaseResponseInterface
    {
        if (is_string($queryString)) {
            parse_str($queryString, $queryString);
        }
        extract($queryString);
        unset($queryString);
        return $this->executeAuth(__FUNCTION__, [], get_defined_vars());
    }

    public function managerExport($queryString = null): BaseResponseInterface
    {
        if (is_string($queryString)) {
            parse_str($queryString, $queryString);
        }
        extract($queryString);
        unset($queryString);
        return $this->executeAuth(__FUNCTION__, [], get_defined_vars());
    }

    /***********************Failed Jobs****************************/
    public function getFailedJobs($start = null, $end = null, $failed_queue = null, $page = 1)
    {
        return $this->executeAuth(__FUNCTION__, [], compact('start', 'end', 'failed_queue', 'page'));
    }

    public function getFailedJob($id)
    {
        return $this->executeAuth(__FUNCTION__, compact('id'));
    }

    public function getFailedJobsQueues()
    {
        return $this->executeAuth(__FUNCTION__, [], get_defined_vars());
    }

    public function retryFailedJobs($ids)
    {
        return $this->executeAuth(__FUNCTION__, [], get_defined_vars());
    }

    public function deleteFailedJobs($ids)
    {
        return $this->executeAuth(__FUNCTION__, [], get_defined_vars());
    }
}
