<?php

namespace Intergo\SubscriptionClient;

use Intergo\BaseClient\Responses\BaseResponseInterface;

abstract class AbstractSubscriptionService
{
    protected SubscriptionClient $client;

    /**
     * AbstractSubscriptionService constructor.
     * https://intergo.atlassian.net/wiki/spaces/ES/pages/2149187587/Communication+between+microservices
     *
     * @param $baseUrl
     * @param $traceId
     */
    public function __construct($baseUrl, $traceId)
    {
        $this->client = new SubscriptionClient($baseUrl, $traceId);
    }

    /**
     * @param $endpoint
     * @param $bodyParams
     * @param array $urlParams
     * @return BaseResponseInterface
     */
    protected function execute($endpoint, $bodyParams, array $urlParams = [])
    {
        $result = $this->client->execute($endpoint, $urlParams, $bodyParams);
        return $this->response($endpoint, $result);
    }

    /**
     * @param $endpoint
     * @param $token
     * @param $bodyParams
     * @param array $urlParams
     * @return BaseResponseInterface
     */
    protected function executeAuth($endpoint, $token, $bodyParams, array $urlParams = [])
    {
        $result = $this->client->executeAuth($endpoint, $token, $urlParams, $bodyParams);
        return $this->response($endpoint, $result);
    }

    /**
     * @param string $action
     * @param BaseResponseInterface $result
     * @return BaseResponseInterface
     */
    abstract public function response(string $action, BaseResponseInterface $result);

    /**
     * @return BaseResponseInterface
     */
    public function getPlans($queryString = [])
    {
        if (is_string($queryString)) {
            parse_str($queryString, $queryString);
        }

        extract($queryString);
        unset($queryString);
        return $this->execute(__FUNCTION__, get_defined_vars());
    }

    /**
     * @param $id
     * @return BaseResponseInterface
     */
    public function getPlanById($id)
    {
        return $this->execute(__FUNCTION__, [], compact('id'));
    }

    /**
     * @param $attributes
     * @return BaseResponseInterface
     */
    public function createPlan($attributes)
    {
        extract($attributes);
        unset($attributes);
        return $this->execute(__FUNCTION__, get_defined_vars());
    }

    /**
     * @param $id
     * @param $attributes
     * @return BaseResponseInterface
     */
    public function updatePlan($id, $attributes)
    {
        extract($attributes);
        unset($attributes);
        return $this->execute(__FUNCTION__, get_defined_vars(), compact('id'));
    }

    /**
     * @param $id
     * @return BaseResponseInterface
     */
    public function deletePlan($id)
    {
        return $this->execute(__FUNCTION__, [], compact('id'));
    }

    /**
     * @param $a2_codes
     * @return BaseResponseInterface
     */
    public function getRentingPlansByParams($a2_codes)
    {
        return $this->execute(__FUNCTION__, compact('a2_codes'));
    }

    /**
     * @param $queryString
     * @return BaseResponseInterface
     */
    public function getSubscriptions($queryString)
    {
        if (is_string($queryString)) {
            parse_str($queryString, $queryString);
        }

        extract($queryString);
        unset($queryString);

        return $this->execute(__FUNCTION__, get_defined_vars());
    }

    /**
     * @param $queryString
     * @return BaseResponseInterface
     */
    public function getServiceSubscriptions($queryString)
    {
        if (is_string($queryString)) {
            parse_str($queryString, $queryString);
        }

        extract($queryString);
        unset($queryString);

        return $this->execute(__FUNCTION__, get_defined_vars());
    }

    /**
     * @param $user_id
     * @param $plan_id
     * @return BaseResponseInterface
     */
    public function storeSubscription($user_id, $plan_id)
    {
        return $this->execute(__FUNCTION__, get_defined_vars());
    }

    /**
     * @param $id
     * @return BaseResponseInterface
     */
    public function getSubscriptionById($id)
    {
        return $this->execute(__FUNCTION__, [], compact('id'));
    }

    /**
     * @param $id
     * @param bool $only_ids
     * @return BaseResponseInterface
     */
    public function getUserSubscriptions($id, bool $only_ids = false)
    {
        return $this->execute(__FUNCTION__, compact('only_ids'), compact('id'));
    }

    /**
     * @param $id
     * @return BaseResponseInterface
     */
    public function getUserSubscriptionsReport($id)
    {
        return $this->execute(__FUNCTION__, [], compact('id'));
    }

    /**
     * @param $ids
     * @return BaseResponseInterface
     */
    public function getSubscriptionsByIds($ids)
    {
        return $this->execute(__FUNCTION__, compact('ids'));
    }

    /**
     * @param $id
     * @return BaseResponseInterface
     */
    public function getSubscriptionBillingLogs($id)
    {
        return $this->execute(__FUNCTION__, [], compact('id'));
    }

    /**
     * @param $id
     * @param $log_id
     * @return BaseResponseInterface
     */
    public function getSubscriptionBillingLog($id, $log_id)
    {
        return $this->execute(__FUNCTION__, [], compact('id', 'log_id'));
    }

    /**
     * @param $id
     * @param $client_cost_in_eur
     * @param $ip_address
     * @param $action
     * @return BaseResponseInterface
     */
    public function storeSubscriptionBillingLog($id, $client_cost_in_eur, $ip_address, $action)
    {
        return $this->execute(__FUNCTION__, compact('client_cost_in_eur', 'ip_address', 'action'), compact('id'));
    }

    /**
     * @param $ids
     * @return BaseResponseInterface
     */
    public function getPlansByIds($ids)
    {
        return $this->execute(__FUNCTION__, compact('ids'));
    }

    /**
     * @return BaseResponseInterface
     */
    public function getPlanCountries()
    {
        return $this->execute(__FUNCTION__, []);
    }

    /**
     * @param $id
     * @return BaseResponseInterface
     */
    public function cancelSubscription($id)
    {
        return $this->execute(__FUNCTION__, [], compact('id'));
    }

    /**
     * @param $id
     * @return BaseResponseInterface
     */
    public function revertCancelSubscription($id)
    {
        return $this->execute(__FUNCTION__, [], compact('id'));
    }

    /**
     * @param $ids
     * @return BaseResponseInterface
     */
    public function checkIfCancelled($ids)
    {
        return $this->execute(__FUNCTION__, compact('ids'));
    }

    /**
     * @param $id
     * @return BaseResponseInterface
     */
    public function renewSubscription($id)
    {
        return $this->execute(__FUNCTION__, [], compact('id'));
    }
}