<?php

namespace Intergo\Shortlinks;

use Intergo\Shortlinks\Enums\ShortlinksEnum;
use Illuminate\Support\Facades\DB;
use Illuminate\Support\Facades\Redis;
use Intergo\Shortlinks\ValueObjects\ContactPlaceholdersVO;

abstract class AbstractShortlinksService
{
    private Redis $redis;
    private ShortlinksRepository $repository;

    /**
     * AbstractWalletService constructor.
     */
    public function __construct(Redis $redis, DB $db)
    {
        $this->redis = $redis;
        $this->repository = new ShortlinksRepository($db, $redis, $this->getConnectionName());
    }

    abstract function log($message, $params);

    /**
     * @return string
     */
    protected function getConnectionName()
    {
        return ShortlinksEnum::APP_DB_CONNECTION;
    }

    /**
     * @param $authUserId
     * @param $message
     * @param $phone
     * @return mixed|string
     */
    public function replaceUniqueShortlinks($authUserId, $message, $phone,$campaignTrackingId = null)
    {
        $uniqueShortlinks = $this->getUniqueShortlinks($authUserId, $message);
        if (!empty($uniqueShortlinks)) {
            $message = $this->replaceShortlinks($uniqueShortlinks, $message, $phone,$campaignTrackingId);
        }
        return $message;
    }

    /**
     * Check if unique shortlink exists
     *
     * @param $authUserId
     * @param $message
     *
     * @return array
     */
    public function getUniqueShortlinks($authUserId, $message)
    {
        $shortlinks = $this->repository->getUserShortlinks($authUserId);
        return array_filter($shortlinks, function ($shortlink) use ($message) {
            return ($shortlink['unique'] && (strpos($message, $shortlink['shortlink']) !== false));
        });
    }

    /**
     * @param array $uniqueShortlinks
     * @param $message
     * @param $phone
     * @return string
     */
    public function replaceShortlinks(array $uniqueShortlinks, $message, $phone, string $campaignTrackingId = null)
    {
        foreach ($uniqueShortlinks as $shortlink) {
            $newShortlink = $this->createFromParent($shortlink['id'], $phone, $campaignTrackingId);
            $message = str_replace($shortlink['shortlink'], $newShortlink, $message);
        }
        return $message;
    }

    /**
     * @param $shortlinkId
     * @param $phone
     * @return string
     */
    public function createFromParent($shortlinkId, $phone, $campaignTrackingId = null)
    {
        $newCode = $this->generateShortlink(7);
        $cacheKey = $this->getCacheKeyForChild($shortlinkId, $newCode);
        $value = ['phone' => $phone, 'parent' => $shortlinkId];

        if (null != $campaignTrackingId) {
            $value['campaign_tracking_id'] = $campaignTrackingId;
        }

        $this->log("Send Cycle - Prepare Unique Shortlink",
            ['shortlink_code' => $newCode, 'value' => $value, 'campaign_tracking_id' => $campaignTrackingId]);

        $this->redis::connection(ShortlinksEnum::CACHE_CONNECTION)->set($cacheKey, serialize($value));
        return $newCode;
    }

    /**
     * @param $authUserId
     * @param int $totalValidMessages
     * @param string|null $message
     * @param string|null $viberTargetUrl
     * @return void
     */
    public function updateShortlinksMessageCount($authUserId, int $totalValidMessages, ?string $message, ?string $viberTargetUrl)
    {
        $shortlinks = $this->repository->getUserShortlinks($authUserId);
        foreach ($shortlinks as $shortlink) {
            if ((strpos($message, $shortlink['shortlink']) !== false) || (strpos($viberTargetUrl, $shortlink['shortlink']) !== false)) {
                $message_count = $shortlink['message_count'] + $totalValidMessages;
                $this->repository->updateShortlink($shortlink['id'], compact('message_count'));
            }
        }
    }

    /**
     * @param $parentId
     * @param $shortlink
     * @return string
     */
    public function getCacheKeyForChild($parentId, $shortlink)
    {
        return "Shortlink:parent:$parentId:shortlink:$shortlink";
    }

    /**
     * @param int $length
     * @return string
     */
    public function generateShortlink(int $length = 5)
    {
        $characters = 'ABCDEFGHIJKLMNOPQRSTUVWXYZabcdefghijklmnopqrstuvwxyz0123456789';
        for (
            $s = '', $i = 0, $z = strlen($characters) - 1; $i != $length; $x = rand(0, $z), $s .= $characters[$x], $i++
        ) {
        }
        return $s;
    }

    /**
     * Replace placeholders.
     *
     * @param ContactPlaceholdersVO $contact
     * @param $msgContent
     *
     * @return array
     */
    public function replacePlaceholders(ContactPlaceholdersVO $contact, $msgContent)
    {
        return str_replace(
            ShortlinksEnum::CONTACT_PLACEHOLDERS,
            [
                $contact->firstname ?? '',
                $contact->lastname ?? '',
                $contact->email ?? '',
                $contact->phone ?? '',
                $contact->custom1 ?? '',
                $contact->custom2 ?? ''
            ],
            $msgContent
        );
    }

    /**
     * Check if the message contains a contact placeholder
     *
     * @param $msgContent
     *
     * @return bool
     */
    public function containsContactPlaceholders($msgContent)
    {
        foreach (ShortlinksEnum::CONTACT_PLACEHOLDERS as $placeholder) {
            if (strpos($msgContent, $placeholder) !== false) {
                return true;
            }
        }
        return false;
    }
}
