<?php

namespace Intergo\ShortlinkClient;

use Intergo\BaseClient\Responses\BaseResponseInterface;
use Intergo\ShortlinkClient\Enums\ShortlinksEnum;
use Intergo\ShortlinkClient\ValueObjects\ContactPlaceholdersVO;
use Intergo\ShortlinkClient\ValueObjects\TrackClickVO;

abstract class AbstractShortlinkService
{
    protected ShortlinkClient $client;

    /**
     * AbstractSubscriptionService constructor.
     * https://intergo.atlassian.net/wiki/spaces/ES/pages/2149187587/Communication+between+microservices
     *
     * @param $baseUrl
     * @param $traceId
     */
    public function __construct($baseUrl, $traceId)
    {
        $this->client = new ShortlinkClient($baseUrl, $traceId);
    }


    /**
     * @param $attributes
     * @return  BaseResponseInterface
     */
    public function createShortlink($attributes)
    {
        extract($attributes);
        unset($attributes);
        return $this->execute(__FUNCTION__, get_defined_vars());
    }


    /**
     * Delete a shortlink.
     *
     * @param int $userId The ID of the logged-in user.
     * @param int $id The ID of the shortlink to be deleted.
     * @return BaseResponseInterface
     */
    public function deleteShortlink(int $userId, int $id)
    {
        return $this->execute(__FUNCTION__, ['user_id' => $userId], compact('id'));
    }
    /**
     * Restore's Soft-deleted shortlink
     * @param $id
     * @return BaseResponseInterface
     */
    public function restoreShortlink(int $userId, $id)
    {
        return $this->execute(__FUNCTION__, ['user_id' => $userId], compact('id'));

    }

    /**
     * Get shortlinks by user.
     *
     * @param int $userId The ID of the user whose shortlinks are being retrieved.
     * @param int $page The current page number for pagination.
     * @param int|null $limit The number of records per page (default to pagination limit).
     * @param string|null $sort The sorting criteria (e.g., column name or sort direction).
     * @param bool|null $onlyTrashed Fetch only the soft-deleted
     * @return BaseResponseInterface
     */
    public function getShortlinksByUser(
        int $userId,
        int $page = 1,
        ?int $limit = null,
        ?string $sort = null,
        ?bool $onlyTrashed
    ) {
        $bodyParams = [
            'page' => $page,
            'limit' => $limit,
            'sort' => $sort,
            'only_trashed' => $onlyTrashed
        ];

        return $this->execute(__FUNCTION__, $bodyParams, compact('userId'));
    }

    /**
     * Get all shortlinks by user without pagination.
     *
     * @param int $userId The ID of the user whose shortlinks are being retrieved.
     * @param string|null $sort The sorting criteria (e.g., column name or sort direction).
     * @param bool|null $onlyTrashed Fetch only the soft-deleted shortlinks.
     * @return BaseResponseInterface
     */
    public function getAllShortlinksByUser(
        int $userId,
        ?string $sort = null,
        ?bool $onlyTrashed = false
    ) {
        $bodyParams = [
            'sort' => $sort,
            'only_trashed' => $onlyTrashed,
        ];

        return $this->execute(__FUNCTION__, $bodyParams, compact('userId'));
    }

    /**
     * Retrieves the details (clicks) of a specific shortlink.
     *
     * @param int $id The ID of the shortlink.
     * @param int $userId The ID of the user requesting the details.
     * @param int $page The current page number for pagination.
     * @param int|null $limit The number of records per page (default to pagination limit).
     * @param string|null $device Filter for a specific device.
     * @param string|null $createdFrom Filter for details created after a specific date.
     * @param string|null $createdTo Filter for details created before a specific date.
     * @return BaseResponseInterface
     */
    public function getShortlinkDetails(
        int $id,
        int $userId,
        int $page = 1,
        ?int $limit = null,
        ?string $device = null,
        ?string $createdFrom = null,
        ?string $createdTo = null
    ) {
        $bodyParams = [
            'user_id' => $userId,
            'page' => $page,
            'limit' => $limit,
            'device' => $device,
            'detail_created_at_from' => $createdFrom,
            'detail_created_at_to' => $createdTo,
        ];

        return $this->execute(__FUNCTION__, $bodyParams, compact('id'));
    }
    /**
     * Retrieves ALL the details (clicks) of a specific shortlink without PAGINATION
     *
     * @param int $id The ID of the shortlink.
     * @param int $userId The ID of the user requesting the details.
     * @param int $page The current page number for pagination.
     * @param bool $all Get all details of shortlink without pagination (used on csv export details)
     * @param int|null $limit The number of records per page (default to pagination limit).
     * @param string|null $device Filter for a specific device.
     * @param string|null $createdFrom Filter for details created after a specific date.
     * @param string|null $createdTo Filter for details created before a specific date.
     * @return BaseResponseInterface
     */
    public function getAllShortlinkDetails(
        int $id,
        int $userId,
        ?string $device = null,
        ?string $createdFrom = null,
        ?string $createdTo = null
    ) {
        $bodyParams = [
            'user_id' => $userId,
            'device' => $device,
            'detail_created_at_from' => $createdFrom,
            'detail_created_at_to' => $createdTo,
        ];

        return $this->execute(__FUNCTION__, $bodyParams, compact('id'));
    }



    /**
     * Updates the message count for a user's shortlinks if they are detected in the provided message or URL.
     *
     * @param int $userId The ID of the user whose shortlinks need to be updated.
     * @param int $totalValidMessages The total number of valid messages to be updated.
     * @param string $message The message or URL to detect shortlinks in.
     * @return void
     */
    public function updateMessageCount(int $userId, int $totalValidMessages, string $message)
    {
        $bodyParams = [
            'user_id' => $userId,
            'total_valid_messages' => $totalValidMessages,
            'message' => $message,
        ];

        return $this->execute(__FUNCTION__, $bodyParams);
    }

    /**
     * Replace shortlink parent unique with a generated (shortliks_unique)table.
     *
     * @param int $userId
     * @param string $campaignTrackingId
     * @param string $recipientPhone
     * @param int $recipientsCount
     * @param string|null $viberTargetUrl Optional parameter
     * @return BaseResponseInterface
     */
    public function replaceUnique(
        int $userId,
        string $message,
        ?string $campaignTrackingId = null,
        string $recipientPhone,
        int $recipientsCount,
        ?string $viberTargetUrl = null
    ) {
        $attributes = [
            'user_id' => $userId,
            'campaign_tracking_id' => $campaignTrackingId,
            'recipient_phone' => $recipientPhone,
            'recipients_count' => $recipientsCount,
            'message' => $message
        ];

        if ($viberTargetUrl !== null) {
            $attributes['viber_target_url'] = $viberTargetUrl;
        }

        return $this->execute(__FUNCTION__, $attributes);
    }

    /**
     * Retrieves all shortlinks for the specified users with pagination.
     *
     * @param array $userIds An array of user IDs for which shortlinks are to be retrieved.
     * @param int $page The page number for pagination.
     *
     * @return BaseResponseInterface The response containing the list of shortlinks.
     */
    public function getAllShortlinks(array $userIds , int $page): BaseResponseInterface
    {
        $bodyParams = [
            'user_id' => $userIds,
            'page' => $page,
        ];

        return $this->execute(__FUNCTION__, $bodyParams);
    }


    /**
     * Updates the status of a specific shortlink.
     *
     * @param string $status The new status to be applied to the shortlink (e.g., 'active', 'disabled').
     * @param int|string $id The ID of the shortlink to be updated.
     *
     * @return BaseResponseInterface The response indicating the result of the update operation.
     */
    public function updateShortlinkStatus(string $status, $id): BaseResponseInterface
    {
        $bodyParams = [
            'status' => $status,
        ];

        return $this->execute(__FUNCTION__, $bodyParams, compact('id'));
    }

    /**
     * Bulk update the status of shortlinks.
     *
     * @param string $status The status to set (e.g., active, disabled).
     * @param array $ids Array of shortlink IDs to update.
     * @return BaseResponseInterface
     */
    public function bulkUpdateStatus(string $status, array $ids)
    {
        $attributes = [
            'status' => $status,
            'ids' => $ids,
        ];

        return $this->execute(__FUNCTION__, $attributes);
    }

    /**
     * Updates the user's shortlink domain.
     *
     * @param string $domain Contains the domain to be updated.
     * @param int $userId The ID of the user whose domain is being updated.
     * @return BaseResponseInterface
     */
    public function updateUserDomain(array $domain, int $userId): BaseResponseInterface
    {
        $bodyParams = [
            'domain' => $domain,
        ];

        return $this->execute(__FUNCTION__, $bodyParams, compact('userId'));
    }

    /**
     * Updates the general shortlink domain. (custom_domains table)
     *
     * @param string $domain Contains the domain to be updated.
     * @return BaseResponseInterface
     */
    public function updateCustomDomain(string $domain): BaseResponseInterface
    {
        $bodyParams = [
            'domain' => $domain
        ];

        return $this->execute(__FUNCTION__, $bodyParams);
    }

    /**
     *
     * Retrieves all the custom domains )
     * @return BaseResponseInterface
     */
    public function getCustomDomains()
    {
        return $this->execute(__FUNCTION__,[]);
    }
    /**
     * Tracks a click on a shortlink.
     *
     * This method sends click details, such as the user's IP, device, operating system, browser,
     * and language, to the shortlink service for analytics purposes.
     *
     * @param TrackClickVO $trackClickVO The value object containing click tracking details.
     * @param string $code The unique code of the shortlink being tracked.
     *
     * @return BaseResponseInterface The response from the shortlink service.
     */
    public function trackShortlinkClick(TrackClickVO $trackClickVO, string $code)
    {
        $bodyParams = $trackClickVO->toArray();
        return $this->execute(__FUNCTION__, $bodyParams, compact('code'));
    }

    /*PLACEHOLDERS METHODS*/
    /**
     * Replace placeholders.
     *
     * @param ContactPlaceholdersVO $contact
     * @param $msgContent
     *
     * @return array
     */
    public function replacePlaceholders(ContactPlaceholdersVO $contact, $msgContent)
    {
        return str_replace(
            ShortlinksEnum::CONTACT_PLACEHOLDERS,
            [
                $contact->firstname ?? '',
                $contact->lastname ?? '',
                $contact->email ?? '',
                $contact->phone ?? '',
                $contact->custom1 ?? '',
                $contact->custom2 ?? ''
            ],
            $msgContent
        );
    }

    /**
     * Check if the message contains a contact placeholder
     *
     * @param $msgContent
     *
     * @return bool
     */
    public function containsContactPlaceholders($msgContent)
    {
        foreach (ShortlinksEnum::CONTACT_PLACEHOLDERS as $placeholder) {
            if (strpos($msgContent, $placeholder) !== false) {
                return true;
            }
        }
        return false;
    }

    /**
     * @param $endpoint
     * @param $bodyParams
     * @param array $urlParams
     * @return BaseResponseInterface
     */
    protected function execute($endpoint, $bodyParams, array $urlParams = [])
    {
        $result = $this->client->execute($endpoint, $urlParams, $bodyParams);
        return $this->response($endpoint, $result);
    }

    abstract public function response(string $action, BaseResponseInterface $result);


}