<?php

namespace Intergo\ShortlinkClient;

use Intergo\BaseClient\Responses\BaseResponseInterface;

abstract class AbstractShortlinkService
{
    protected ShortlinkClient $client;

    /**
     * AbstractSubscriptionService constructor.
     * https://intergo.atlassian.net/wiki/spaces/ES/pages/2149187587/Communication+between+microservices
     *
     * @param $baseUrl
     * @param $traceId
     */
    public function __construct($baseUrl, $traceId)
    {
        $this->client = new ShortlinkClient($baseUrl, $traceId);
    }


    /**
     * @param $attributes
     * @return  BaseResponseInterface
     */
    public function createShortlink($attributes)
    {
        extract($attributes);
        unset($attributes);
        return $this->execute(__FUNCTION__, get_defined_vars());
    }


    /**
     * Delete a shortlink.
     *
     * @param int $userId The ID of the logged-in user.
     * @param int $id The ID of the shortlink to be deleted.
     * @return BaseResponseInterface
     */
    public function deleteShortlink(int $userId, int $id)
    {
        return $this->execute(__FUNCTION__, ['user_id' => $userId], compact('id'));
    }
    /**
     * Restore's Soft-deleted shortlink
     * @param $id
     * @return BaseResponseInterface
     */
    public function restoreShortlink(int $userId, $id)
    {
        return $this->execute(__FUNCTION__, ['user_id' => $userId], compact('id'));

    }
    /**
     * Get shortlinks by user.
     *
     * @param int $userId The ID of the user whose shortlinks are being retrieved.
     * @param int $page The current page number for pagination.
     * @param int|null $limit The number of records per page (default to pagination limit).
     * @param string|null $sort The sorting criteria (e.g., column name or sort direction).
     * @return BaseResponseInterface
     */
    public function getShortlinksByUser(int $userId, int $page = 1, ?int $limit = null, ?string $sort = null)
    {
        $bodyParams = [
            'page' => $page,
            'limit' => $limit,
            'sort' => $sort,
        ];

        return $this->execute(__FUNCTION__, $bodyParams, compact('userId'));
    }

    /**
     * Retrieves the details (clicks) of a specific shortlink.
     *
     * @param int $id The ID of the shortlink.
     * @param int $userId The ID of the user requesting the details.
     * @param int $page The current page number for pagination.
     * @param int|null $limit The number of records per page (default to pagination limit).
     * @param string|null $device Filter for a specific device.
     * @param string|null $createdFrom Filter for details created after a specific date.
     * @param string|null $createdTo Filter for details created before a specific date.
     * @return BaseResponseInterface
     */
    public function getShortlinkDetails(
        int $id,
        int $userId,
        int $page = 1,
        ?int $limit = null,
        ?string $device = null,
        ?string $createdFrom = null,
        ?string $createdTo = null
    ) {
        $bodyParams = [
            'user_id' => $userId,
            'page' => $page,
            'limit' => $limit,
            'device' => $device,
            'detail_created_at_from' => $createdFrom,
            'detail_created_at_to' => $createdTo,
        ];

        return $this->execute(__FUNCTION__, $bodyParams, compact('id'));
    }


    /**
     * Updates the message count for a user's shortlinks if they are detected in the provided message or URL.
     *
     * @param int $userId The ID of the user whose shortlinks need to be updated.
     * @param int $totalValidMessages The total number of valid messages to be updated.
     * @param string $message The message or URL to detect shortlinks in.
     * @return void
     */
    public function updateMessageCount(int $userId, int $totalValidMessages, string $message)
    {
        $bodyParams = [
            'user_id' => $userId,
            'total_valid_messages' => $totalValidMessages,
            'message' => $message,
        ];

        return $this->execute(__FUNCTION__, $bodyParams);
    }

    /**
     * Replace shortlink parent unique with a generated (shortliks_unique)table.
     *
     * @param int $userId
     * @param string $campaignTrackingId
     * @param string $recipientPhone
     * @param int $recipientsCount
     * @param string|null $viberTargetUrl Optional parameter
     * @return BaseResponseInterface
     */
    public function replaceUnique(
        int $userId,
        string $message,
        string $campaignTrackingId,
        string $recipientPhone,
        int $recipientsCount,
        ?string $viberTargetUrl = null
    ) {
        $attributes = [
            'user_id' => $userId,
            'campaign_tracking_id' => $campaignTrackingId,
            'recipient_phone' => $recipientPhone,
            'recipients_count' => $recipientsCount,
            'message' => $message
        ];

        if ($viberTargetUrl !== null) {
            $attributes['viber_target_url'] = $viberTargetUrl;
        }

        return $this->execute(__FUNCTION__, $attributes);
    }

    public function getAllShortlinks($attributes)
    {
        return $this->execute(__FUNCTION__, $attributes);
    }

    public function updateShortlinkStatus($attributes, $id)
    {
        return $this->execute(__FUNCTION__, $attributes, compact('id'));
    }

    public function bulkUpdateStatus($attributes)
    {
        return $this->execute(__FUNCTION__, $attributes);
    }

    public function updateUserDomain($attributes, $userId)
    {
        return $this->execute(__FUNCTION__, $attributes, compact('userId'));
    }
    public function updateCustomDomain($attributes)
    {
        return $this->execute(__FUNCTION__, $attributes);
    }
    public function getCustomDomains($attributes)
    {
        return $this->execute(__FUNCTION__,$attributes);
    }


    /**
     * @param $endpoint
     * @param $bodyParams
     * @param array $urlParams
     * @return BaseResponseInterface
     */
    protected function execute($endpoint, $bodyParams, array $urlParams = [])
    {
        $result = $this->client->execute($endpoint, $urlParams, $bodyParams);
        return $this->response($endpoint, $result);
    }

    abstract public function response(string $action, BaseResponseInterface $result);


}