<?php

namespace Intergo\Providers\MessageManager\Drivers;

use Intergo\Providers\Exceptions\CustomException;
use Intergo\Providers\MessageManager\Exceptions\SMSException;
use Illuminate\Http\Request;
use ReflectionClass;

abstract class Driver
{
    /**
     * The sender of the message.
     *
     * @var string
     */
    protected $sender;

    /**
     * The recipient of the message.
     *
     * @var string|array
     */
    protected $recipient;

    /**
     * The message to send.
     *
     * @var string
     */
    public $message;

    /**
     * Array of number and message to send
     *
     * @var array
     */
    protected $messages;

    /**
     * Config for Drivers
     *
     * @var object
     */
    protected $config;

    /**
     * Identify Campaign
     *
     * @var bool
     */
    protected $isCampaign;

    /**
     * Transcode message
     *
     * @var bool
     */
    protected $isTranscoded = true;

    /**
     * Name of the driver
     *
     * @var string
     */
    public $name;

    /**
     * Array with request data
     *
     * @var array $request
     */
    protected $request;

    public function __construct()
    {
        $this->sender = 'SMSTo';
        $name = (new ReflectionClass($this))->getShortName();
        $this->name = str_replace('Driver', '', $name);
    }


    /**
     * If the message is Flash SMS
     *
     * @var bool
     */
    protected $isFlashSMS = false;

    /**
     * Sending message to recipients via different drivers(providers)
     *
     * {@inheritdoc}
     */
    abstract public function send();

    /**
     * @return string
     */
    abstract public function getMessageField();


    /**
     * Receive callbacks from Provider
     *
     * @param $data
     * @return array
     * @throws CustomException
     */
    abstract public function receiveCallback($data);

    /**
     * After sending message provider(provider have different fields in response), I want to map different responses to have standard mapping
     * With this function, I will be getting standardized fields for all providers
     *
     * {@inheritdoc}
     */
    abstract public function mapper(array $resultMessage);

    /**
     * Set the sender of the message.
     *
     * @param object $config
     *
     * @return $this
     * @throws \Throwable
     */
    public function config($config)
    {
        throw_if(empty($config), SMSException::class, 'Sender cannot be empty');
        $this->config = $config;
        return $this;
    }

    /**
     * Set the sender of the message.
     *
     * @param string $sender
     *
     * @return $this
     * @throws \Throwable
     */
    public function from(string $sender = 'SMSTo')
    {
        $this->sender = $sender;
        return $this;
    }

    /**
     * Set the multiple message message.
     *
     * @param array $messages
     *
     * @return $this
     * @throws \Throwable
     */
    public function messages($messages)
    {
        throw_if(empty($messages), SMSException::class, 'Message cannot be empty');
        $this->messages = $messages;
        return $this;
    }

    /**
     * Set the recipient of the message.
     *
     * @param string|array $recipient
     *
     * @return $this
     * @throws \Throwable
     */
    public function to($recipient)
    {
        throw_if(is_null($recipient), SMSException::class, 'Recipients cannot be empty');
        $this->recipient = $recipient;
        return $this;
    }

    /**
     * Set the recipient of the message.
     *
     * @param bool $isCampaign
     *
     * @return $this
     */
    public function isCampaign(bool $isCampaign)
    {
        $this->isCampaign = $isCampaign;
        return $this;
    }

    /**
     * Check if message is transcoded
     *
     * @param bool $isTranscoded
     *
     * @return $this
     */
    public function isTranscoded(bool $isTranscoded)
    {
        $this->isTranscoded = $isTranscoded;
        return $this;
    }

    /**
     * Set the content of the message.
     *
     * @param string $message
     *
     * @return $this
     * @throws \Throwable
     */
    public function message(string $message)
    {
        throw_if(empty($message), SMSException::class, 'Message text is required');
        $this->message = $message;
        return $this;
    }

    /**
     * Set the extra request data if needed
     *
     * @param array $requestData
     *
     * @return $this
     * @throws \Throwable
     */
    public function requestData($requestData)
    {
        return $this;
    }

    /**
     * Set if the message is Flash SMS
     *
     * @param string $message
     *
     * @return $this
     * @throws \Throwable
     */
    public function isFlashSMS(bool $isFlashSMS)
    {
        $this->isFlashSMS = $isFlashSMS;
        return $this;
    }
}
