<?php

namespace Intergo\Providers\MessageManager\Drivers;

use Intergo\Providers\Helpers\StringHelper;
use Intergo\Log\GrayLogHandler;
use Intergo\Providers\Enums\LoggingTagEnum;
use Intergo\Providers\Enums\ViberCommonStatusEnum;
use Intergo\Providers\Exceptions\CustomException;
use GuzzleHttp\Client;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Cache;
use Intergo\Providers\Enums\FailedReasonEnum;
use Throwable;

class NthViberDriver extends Driver
{
    const DELIVERED_STATUSES = [
        'DELIVERED',
        'SEEN',
        'SUBMITTED'
    ];
    const FAILED_STATUSES = [
        'DELIVERY_FAILED',
        'BLACKLISTED',
        'EXPIRED',
    ];
    const REJECTED_STATUSES = [
        'ERROR',
        'INSUFFICIENT_FUNDS',
        'NO_ROUTE', 'REJECTED',
        'THROTTLING',
    ];
    const REJECTED_ERROR_CODES = [
        ViberCommonStatusEnum::NOT_VIBER_USER,
        ViberCommonStatusEnum::NO_SUITABLE_DEVICE,
        self::TIMEOUT,
        self::INVALID_PHONE_NUMBER_FORMAT,
        self::REQUEST_ERROR,
        self::INTERNAL_ERROR
    ];

    const UNSUBSCRIBED_ERROR_CODES = [
        ViberCommonStatusEnum::USER_BLOCKED,
    ];

    //CONSTS FOR VIBER ERRORS
    const	TIMEOUT = 'TIMEOUT';
    const	INVALID_PHONE_NUMBER_FORMAT	= "INVALID_PHONE_NUMBER_FORMAT";
    const	REQUEST_ERROR = 'REQUEST_ERROR';
    const	INTERNAL_ERROR = 'INTERNAL_ERROR';

    /**
     * Send messages
     *
     * @author Panayiotis Halouvas <phalouvas@kainotomo.com>
     * @link https://developers.nth.ch/channel/
     *
     * @return array
     */
    public function send()
    {
        $message = $this->messages[0];

        $input = [
            'channels' => ['VIBER'],
            'destinations' => [
                0 => ['phoneNumber' => $message['to']]
            ],
            'messagePriority' => 'NORMAL',
            'requestId' => $message['_id'],
            'transactionId' => $message['_id'],
            'dlr' => true,
            'dlrUrl' => config('services.nth.notify_url'),
            'viber' => [
                'sender' => $message['sender_id'],
                'tag' => 'SMSto',
                'text' => $message['message'],
                'ttl' => 14440,
                'label' => 'promotion',
                'image' => $message['viber_image_url'],
                'buttonCaption' => $message['viber_caption'],
                'buttonAction' => $message['viber_target_url'],
                'smartphoneOnly' => false
            ],
        ];

        return $this->sendRequest($input);
    }

    /**
     * Guzzle HTTP Request for all API Call
     *
     * @author Panayiotis Halouvas <phalouvas@kainotomo.com>
     *
     * @param array $input
     *
     * @return array
     */
    public function sendRequest(array $input)
    {
        $params = $this->messages[0]['viber_params']['nth'];
        $authorizationToken = base64_encode($params['username'] . ':' . $params['password']);
        $client = new Client([
            'headers' => [
                'Content-Type' => 'application/json',
                'Authorization' => 'Basic ' . $authorizationToken
            ]
        ]);
        try{
            $clientResponse = $client->post($params['url'], ['json' => $input]);
            $clientResponseArr = json_decode($clientResponse->getBody(), true);
            $is_accepted = $clientResponseArr['messages'][0]['status']['code'] == 'ACCEPTED';
            $response = array();
            if ($is_accepted) {
                $response['SENT-TO-PROVIDER'][] = [
                    '_id' => $input['transactionId'],
                    'message_id' => $clientResponseArr['messages'][0][$this->getMessageField()] ?? null,
                    'status' =>'SENT-TO-PROVIDER',
                ];
            } else {
                $response['FAILED'][] = [
                    '_id' => $input['transactionId'],
                    'message_id' => $clientResponseArr['messages'][0][$this->getMessageField()] ?? null,
                    'status' =>'FAILED',
                    'error' => true,
                    'reason' => $clientResponseArr['messages'][0]['status']['description'],
                    'data' => json_encode($input),
                ];
            }

        } catch (\Exception $e) {
            $response['REJECTED'][] = [
                '_id' => $input['transactionId'],
                'status' =>'REJECTED',
                'error' => true,
                'reason' => $e->getMessage(),
                'data' => json_encode($input),
            ];
        }

        return $response;
    }

    /**
     * Set the "internal_failed_reason" and "failed_reason".
     *
     * @param array $resultMessage
     * @param string $status_details
     * @return array
     */
    protected function setFailedReasons(array $resultMessage, string $status_details)
    {
        try {
            $resultMessage['internal_failed_reason'] = $status_details;
            $resultMessage['failed_reason'] = FailedReasonEnum::getCustomerFailedReason($status_details);
        } catch (Throwable $exception) {
            app(GrayLogHandler::class)->setTag(LoggingTagEnum::MESSAGES_SERVICE)->error("NthViberDriver@setFailedReasons: {$exception->getMessage()}", [
                'result_message_status' => $resultMessage['status'] ?? null,
                'result_message_id' => $resultMessage['campaignTrackingId'] ?? $resultMessage[$this->getMessageField()] ?? null
            ]);
            $resultMessage['internal_failed_reason'] = "Failed reason cannot be set because of the error in our code: {$exception->getMessage()}";
            $resultMessage['failed_reason'] = FailedReasonEnum::DEFAULT_CUSTOMER_FAILED_REASON;
        }

        return $resultMessage;
    }

    /**
     * After sending message provider(provider have different fields in response), I want to map different responses to have standard mapping
     * With this function, I will be getting standardized fields for all providers
     *
     * @author Panayiotis Halouvas <phalouvas@kainotomo.com>
     *
     * @param array $resultMessage
     * @return array
     */
    public function mapper(array $resultMessage)
    {
        if (!isset($resultMessage[$this->getMessageField()], $resultMessage['transactionId'])) {
            throw new CustomException('Invalid format', 400);
        }
        $status_details = $resultMessage['status']['details'];
        if (in_array($resultMessage['status']['code'], NthViberDriver::DELIVERED_STATUSES)) {
            $resultMessage['status'] = 'DELIVERED';
        } elseif (in_array($resultMessage['status']['code'], NthViberDriver::FAILED_STATUSES)) {
            $resultMessage['status'] = 'FAILED';
            $resultMessage = $this->setFailedReasons($resultMessage,$status_details);

        } elseif (in_array($resultMessage['status']['code'], NthViberDriver::REJECTED_STATUSES)) {
            $resultMessage['status'] = 'REJECTED';
            $resultMessage = $this->setFailedReasons($resultMessage,$status_details);
        } elseif (in_array($resultMessage['status']['code'], NthViberDriver::UNSUBSCRIBED_ERROR_CODES)){
            $resultMessage['status'] = 'UNSUBSCRIBED';
        } else {
            $resultMessage['status'] = 'FAILED';
            $resultMessage = $this->setFailedReasons($resultMessage,$status_details);
        }

        if (StringHelper::strpos_in_array($status_details, NthViberDriver::REJECTED_ERROR_CODES) !== false) {
            $resultMessage['status'] = 'REJECTED';
            $resultMessage = $this->setFailedReasons($resultMessage,$status_details);
        }
        return $resultMessage;
    }

    /**
     * Receive callbacks from Provider
     *
     * @param $data
     * @return array
     * @throws CustomException
     */
    public function receiveCallback($data)
    {
        return $this->mapper($data);
    }

    /**
     * @return string
     */
    public function getMessageField()
    {
        return 'messageId';
    }
}
