<?php

namespace Intergo\Providers\MessageManager\Drivers;

use \Illuminate\Http\Request;
use Intergo\Providers\MessageManager\Exceptions\SMSException;
use App\Models\UserViberSettings;

class RouteeViberDriver extends RouteeSmsDriver
{
    /**
     *
     * @var array Viber defined variables
     */
    protected $requestData;

    public function __construct()
    {
        parent::__construct();
        $this->applicationId = config('services.viber.applicationId');
        $this->secret = config('services.viber.secret');
        $this->endpoint = config('services.viber.api_url');
    }


    /**
     * Set the viber message user defined settings.
     *
     * @author Panayiotis Halouvas <phalouvas@kainotomo.com>
     *
     * @param array $requestData
     *
     * @return $this
     * @throws \Throwable
     */
    public function requestData($requestData)
    {
        throw_if(empty($requestData), SMSException::class, 'Viber settings cannot be empty');
        $transformedRequest = [];
        $transformedRequest['user_id'] = $requestData['user_id'];
        if (isset($requestData['viber_image_url'])) {
            $transformedRequest['viber_image_url'] = $requestData['viber_image_url'];
        }
        if (isset($requestData['viber_target_url'])) {
            $transformedRequest['viber_target_url'] = $requestData['viber_target_url'];
        }
        if (isset($requestData['viber_caption'])) {
            $transformedRequest['viber_caption'] = $requestData['viber_caption'];
        }
        $this->requestData = $transformedRequest;
        return parent::requestData($requestData);
    }

    /**
     * Send multiple messages one message per number
     * Format 1 => MULTI MESSAGE ==> [[to: "+9779856034616", message: "This is test"], [to: "+9779856034617", message: "This is test"]]
     *
     * @author Panayiotis Halouvas <phalouvas@kainotomo.com>
     *
     * @param $messages
     *
     * @return array
     * @throws Exception
     */
    public function sendMessageToRecipient($messages)
    {
        $response = [];
        $postFields = $this->getCommonPostFields();

        foreach ($messages as $message) {
            $postFields['to'] = [$message['to']];
            $postFields['campaignName'] = uniqid('smsto_');
            $postFields['body']['text'] = str_replace('  ', ' ', $message['message']);

            $providerResponse = $this->sendRequest('POST', $this->endpoint, $postFields); // SENT|FAILED
            $providerResponse['_id'] = $message['_id'];
            if (!empty($providerResponse['trackingId']))
            {
                $bulkId = $providerResponse['trackingId'];
                $providerResponse['message_id'] = $bulkId; // Success
            }
            $response[$providerResponse['status']][] = $providerResponse;
        }

        return $response;
    }

    /**
     * Send one message to many numbers
     * Format 3 => CAMPAIGN MESSAGE ==> to: ["+9779856034616", "+9779856034617"], message: "This is test"
     *
     * @author Panayiotis Halouvas <phalouvas@kainotomo.com>
     *
     * @return boolean|array
     */
    public function sendCampaignToRecipient()
    {
        $postFields = $this->getCommonPostFields();

        $postFields['to'] = $this->recipient;
        $postFields['campaignName'] = time() . "-" . str_random(10);
        $postFields['body']['text'] = str_replace('  ', ' ', $this->message);

        $response = $this->sendRequest('POST', $this->endpoint, $postFields);

        $isError = $this->handleIfError($response);

        return $isError ? [false] : [$response];
    }

    /**
     * Get common post fields
     *
     * @author Panayiotis Halouvas <phalouvas@kainotomo.com>
     *
     * @return array
     */
    private function getCommonPostFields() {
        $settings = UserViberSettings::where('user_id', $this->requestData['user_id'])->first();
        $postFields['senderInfoTrackingId'] = '355216b9-d26a-47fc-9470-2c674132bc57';
        if (config('app.env') === 'production')
        {
            $postFields['senderInfoTrackingId'] = $settings->sender_id;
        }

        if (!empty($this->requestData['viber_image_url']))
        {
            $postFields['body']['imageURL'] = $this->requestData['viber_image_url'];
        }
        if (!empty($this->requestData['viber_target_url']) && !empty($this->requestData['viber_caption']))
        {
            $postFields['body']['action']['targetUrl'] = $this->requestData['viber_target_url'];
            $postFields['body']['action']['caption'] = $this->requestData['viber_caption'];
        }
        $postFields['ttl'] = 5000;
        $postFields['callbackUrl'] = config('app.url') . '/sms-webhook/viber';
        $postFields['inboundUrl'] = config('app.url') . '/viber/inbound';
        return $postFields;
    }


    /**
     * After sending message provider(provider have different fields in response), I want to map different responses to have standard mapping
     * With this function, I will be getting standardized fields for all providers
     *
     * @author Panayiotis Halouvas <phalouvas@kainotomo.com>
     *
     * Routee
     * {@inheritdoc}
     */
    public function mapper(array $resultMessage)
    {
        $resultMessage = parent::mapper($resultMessage);
        $resultMessage['provider'] = 'Routee';
        return $resultMessage;
    }

}
