<?php

namespace Intergo\Providers\MessageManager\Drivers;

use Intergo\Providers\Enums\LoggingTagEnum;
use Intergo\Providers\Enums\FailedReasonEnum;
use Intergo\Providers\Exceptions\CustomException;
use Intergo\Log\GrayLogHandler;
use Exception;
use GuzzleHttp\Client;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Cache;
use Intergo\Providers\Enums\MessageEnum;
use Throwable;

/**
 * Common SMS Driver class
 *
 * @author Panayiotis Halouvas <phalouvas@kainotomo.com>
 */
class RouteeSmsDriver extends Driver
{
    /**
     * @var
     */
    protected $from;

    /**
     * Where infobip will send response
     *
     * @var string
     */
    protected $notifyUrl;

    /**
     * @var
     */
    protected $callbackData;

    /**
     *
     * @var string Application ID
     */
    protected $applicationId;

    /**
     *
     * @var string secret key of application
     */
    protected $secret;

    /**
     * @var GrayLogHandler
     */
    private $grayLogger;

    /**
     *
     * @var string URL endpoint
     */
    protected $endpoint;

    public function __construct()
    {
        parent::__construct();
        $this->endpoint = config('services.routee.api_url');
        $this->from = 'SMSTO';
        $this->notifyUrl = config('services.routee.notify_url');
        $this->grayLogger = app(GrayLogHandler::class)->setTag(LoggingTagEnum::MESSAGES_SERVICE);
    }

    /**
     * @return array
     * @throws Exception
     */
    public function send()
    {
        $this->logEvent("Calling method","send");
        // Format 1 => MULTI MESSAGE ==> [[to: "+9779856034616", message: "This is test"], [to: "+9779856034617", message: "This is test"]]
        if (!empty($this->messages))
        {
            $this->logEvent("Sending sms","send");
            return $this->sendMessageToRecipient($this->messages);
        }

        // Format 3 => CAMPAIGN MESSAGE ==> to: ["+9779856034616", "+9779856034617"], message: "This is test"
        return $this->sendCampaignToRecipient();
    }

    /**
     * @param $messages
     *
     * @return array
     * @throws Exception
     */
    public function sendMessageToRecipient($messages)
    {
        $this->logEvent("Calling method",'sendMessageToRecipient');
        $response = [];
        foreach ($messages as $message)
        {
            $text = str_replace('  ', ' ', $message['message']);
            $sender = $this->sender ?? $this->from;
            $postFields = [
                'callback' => [
                    'strategy' => 'OnChange',
                    'url' => $this->notifyUrl,
                ],
                'body' => trim($text),
                'to' => $message['to'],
                'from' => $sender,
                'transcode' => $this->isTranscoded,
                'flash'     => ($this->isFlashSMS) ? TRUE : FALSE
            ];

            $providerResponse = $this->sendRequest('POST', $this->endpoint, $postFields); // SENT|FAILED
            $providerResponse['_id'] = $message['_id'];
            if(!empty($providerResponse['trackingId']))
            {
                $bulkId = $providerResponse['trackingId'];
                $providerResponse['message_id'] = $bulkId; // Success
            }
            $response[$providerResponse['status']][] = $providerResponse;

        }
        $this->logEvent("Finished sending messages",'sendMessageToRecipient','info',[
            'response' => $response
        ]);
        return $response;
    }

    public function sendCampaignToRecipient()
    {
        $this->logEvent("Sending campaigns","sendCampaignToRecipient",'info',[
            'recipients' => $this->recipient
        ]);
        $sender = $this->from;

        $text = str_replace('  ', ' ', $this->message);
        $sender = $sender ? $sender : $this->from;
        $postFields = [
            'callback' => [
                'strategy' => 'OnChange',
                'url' => $this->notifyUrl,
            ],
            'body' => trim($text),
            'to' => implode(',', $this->recipient),
            'from' => $sender,
            'transcode' => $this->isTranscoded,
            'flash'     => ($this->isFlashSMS) ? TRUE : FALSE
        ];

        $response = $this->sendRequest('POST', $this->endpoint . '/campaign', $postFields);

        $isError = $this->handleIfError($response);

        $this->logEvent("Finished sending campaign","sendCampaignToRecipient",'info',[
            'response' => $response
        ]);

        return $isError ? [false] : [$response];
    }

    /**
     * Execute request.
     *
     * @param      $method
     * @param      $endPoint
     * @param null $bodyContent
     *
     * @return mixed
     * @throws Exception
     */
    protected function sendRequest($method, $endPoint, $bodyContent = null)
    {
        $this->logEvent("Sending request","sendRequest");
        $authorizationToken = $this->getAuthorizationToken();
        if (empty($authorizationToken))
        {
            $this->logEvent("Auth token is empty","sendRequest",'error');
            return [
                'error' => true,
                'reason' => 'Invalid Credentials for Routee',
                'data' => null,
                'status' => 'REJECTED'
            ];
        }
        $client = new Client([
            'headers' => [
                'Content-Type' => 'application/json',
                'Authorization' => 'Bearer ' . $authorizationToken
            ]
        ]);
        try{
            $resp = $client->post($endPoint, [
                'json' => $bodyContent
            ]);
            $response = json_decode($resp->getBody()->getContents(), true);

            if(isset($response['developerMessage']))
            {
                $reason = isset($response['properties']) ? json_encode($response['properties']) : $response['developerMessage'];
                $this->logEvent("Received error during send (reason :$reason)","sendRequest",'error');

                $response['error'] = true;
                $response['reason'] = $reason;
                $response['data'] = $bodyContent;
                $response['status'] = 'REJECTED';

                $this->grayLogger->info('Routee Error:' . $reason);
                return $response;
            }
            if(!isset($response['trackingId']))
            {
                $reason = json_encode($response);
                $this->logEvent("Tracking id not set (reason :$reason)","sendRequest",'error');

                $response['error'] = true;
                $response['reason'] = $reason;
                $response['data'] = json_encode($bodyContent);
                $response['status'] = 'REJECTED';

                $this->grayLogger->info('Routee Error:' . $reason);
                return $response;
            }
            $this->logEvent("Send succeeded","sendRequest",'info',[
                'response' => $response
            ]);

            $response['status'] = 'SENT-TO-PROVIDER';
            return $response;
        } catch (\Exception $e) {
            $reason = $e->getMessage();
            $response = [
                'error' => true,
                'reason' => $reason,
                'data' => json_encode($bodyContent),
                'status' => 'REJECTED'
            ];
            $this->logEvent("Received exception (reason :$reason)","sendRequest",'error',[
                'request' => json_encode($bodyContent)
            ]);
            return $response;
        }
        catch (\Throwable $f) {
            $reason = $f->getMessage();
            $response = [
                'error' => true,
                'reason' => $reason,
                'data' => json_encode($bodyContent),
                'status' => 'REJECTED'
            ];
            $this->logEvent("Received throwable (reason :$reason)","sendRequest",'error',[
                'request' => json_encode($bodyContent)
            ]);
            return $response;
        }
    }

    public function prepareResponse($response)
    {
        $response = new \stdClass();
    }

    /**
     * Handle Routee response error if any.
     *
     * @param $response
     *
     * @return bool
     */
    protected function handleIfError($response)
    {
        if (isset($response->developerMessage))
        {
            $this->grayLogger->error("RouteeSmsDriver@handleIfError", [
                'exception_message' => $response->developerMessage ?? 'There is an error response by Routee.',
            ]);

            if (isset($response->errors) && is_array($response->errors))
            {

                foreach ($response->errors as $error)
                {
                    $this->grayLogger->error("RouteeSmsDriver@handleIfError multiple errors", [
                        'exception_message' => $error->developerMessage ?? 'There is an error response by Routee.',
                    ]);
                }
            }

            return true;
        }

        return false;
    }

    /**
     * Receive callbacks from Provider
     *
     * @param $data
     * @return array
     * @throws CustomException
     */
    public function receiveCallback($data)
    {
        return $this->mapper($data);
    }

    /**
     * Get authorization token.
     *
     *
     * @return mixed
     * @throws Exception
     */
    public function getAuthorizationToken()
    {
        $token = base64_encode($this->applicationId . ":" . $this->secret);
        $this->logEvent("Generating token","getAuthorizationToken",'info');

        $key = "routee:token:".$token;

        $this->logEvent("Checking if key is available (key : $key)","getAuthorizationToken",'info');
        if(Cache::has($key)){
            $this->logEvent("Key exist (key : $key)","getAuthorizationToken",'info');
            return Cache::get($key);
        }

        $curl = curl_init();

        curl_setopt_array($curl, [
            CURLOPT_URL => config('services.routee.auth_url'),
            CURLOPT_RETURNTRANSFER => true,
            CURLOPT_ENCODING => '',
            CURLOPT_MAXREDIRS => 10,
            CURLOPT_TIMEOUT => 30,
            CURLOPT_HTTP_VERSION => CURL_HTTP_VERSION_1_1,
            CURLOPT_CUSTOMREQUEST => 'POST',
            CURLOPT_POSTFIELDS => 'grant_type=client_credentials',
            CURLOPT_HTTPHEADER => [
                'authorization: Basic ' . $token,
                'content-type: application/x-www-form-urlencoded',
            ],
        ]);

        $response = curl_exec($curl);
        $err = curl_error($curl);
        curl_close($curl);

        if ($err)
        {
            $this->logEvent("Routee Error while getting Token: $err","getAuthorizationToken",'info');
            return null;
        }

        $response = json_decode($response);

        if (!isset($response->access_token))
        {
            $this->logEvent("Invalid or Credentials Not found! Please add credentials for Routee","getAuthorizationToken",'info');
            return null;
        }

        $access_token = $response->access_token;
        $expiresIn = $response->expires_in ?? 300; //5 minutes
        Cache::put($key,$access_token, $expiresIn);
        $this->logEvent("Setting access token into Cache","getAuthorizationToken",'info');

        return $access_token;
    }

    /**
     * After sending message provider(provider have different fields in response), I want to map different responses to have standard mapping
     * With this function, I will be getting standardized fields for all providers
     *
     * Routee
     * {@inheritdoc}
     */
    public function mapper(array $resultMessage)
    {
        if (!isset($resultMessage['campaignTrackingId']) && !isset($resultMessage[$this->getMessageField()]))
        {
            throw new CustomException('Invalid format', 400);
        }

        $bulkId = $resultMessage['campaignTrackingId'] ?? $resultMessage[$this->getMessageField()];
        $statusArr = $resultMessage['status'];
        $status = strtoupper($resultMessage['status']['name']);
        $to = str_replace('+', '', $resultMessage['to']);
        if(in_array($status, MessageEnum::DELIVERED_STATUSES) && $status !== 'SENT')
        {
            $resultMessage['status'] = 'DELIVERED';
        }
        elseif($status == 'SEEN')
        {
            $resultMessage['status'] = 'DELIVERED';
        }
        elseif(in_array($status,['EXPIRED', 'UNSENT']))
        {
            $resultMessage['status'] = 'REJECTED';
            $resultMessage = $this->setFailedReasons($resultMessage, $statusArr);
        }
        elseif(in_array($status,MessageEnum::FAILED_STATUSES))
        {
            $resultMessage['status'] = 'FAILED';
            $resultMessage = $this->setFailedReasons($resultMessage, $statusArr);
        }
        else
        {
            return false;
        }
        $resultMessage['message_id'] = $bulkId;
        return $resultMessage;
    }

    /**
     * Set the "internal_failed_reason" and "failed_reason".
     *
     * @param array $resultMessage
     * @param array $statusArr
     * @return array
     */
    protected function setFailedReasons(array $resultMessage, array $statusArr)
    {
        try {
            $resultMessage['internal_failed_reason'] = 'name: ' .  $statusArr['name'] . ' detailedStatus: ' . $statusArr['reason']['detailedStatus'] . ' description: ' . $statusArr['reason']['description'];
            $resultMessage['failed_reason'] = FailedReasonEnum::getCustomerFailedReason($statusArr['reason']['detailedStatus']);
        } catch (Throwable $exception) {
            $this->grayLogger->error("RouteeSmsDriver@setFailedReasons: {$exception->getMessage()}", [
                'result_message_status' => $resultMessage['status'] ?? null,
                'result_message_id' => $resultMessage['campaignTrackingId'] ?? $resultMessage[$this->getMessageField()] ?? null
            ]);
            $resultMessage['internal_failed_reason'] = "Failed reason cannot be set because of the error in our code: {$exception->getMessage()}";
            $resultMessage['failed_reason'] = FailedReasonEnum::DEFAULT_CUSTOMER_FAILED_REASON;
        }

        return $resultMessage;
    }

    /**
     * Use this method to record any logs
     * @param $message
     * @param $method
     * @param string $level
     * @param array $data
     */
    private function logEvent($message, $method, $level = "info", array $data = [])
    {
        $extraParams = [
            'driver' => self::class
        ];
        $this->grayLogger->$level(__CLASS__ . " - $method - " . $message, array_merge($data, $extraParams));
    }


    /**
     * @return string
     */
    public function getMessageField()
    {
        return 'messageId';
    }
}
