<?php

namespace Intergo\Providers\MessageManager\Drivers;

use Intergo\Log\GrayLogHandler;
use Intergo\Providers\Enums\FailedReasonEnum;
use Intergo\Providers\Enums\LoggingTagEnum;
use Intergo\Providers\Exceptions\CustomException;
use Intergo\HashHelper\HashHelper;
use GuzzleHttp\Client;
use Throwable;

class InfoBipDriver extends Driver
{
    protected $provider;

    /**
     * @var
     */
    protected $path;

    /**
     * Priority of the message
     *
     * @var int $priority
     */
    protected $priority = 2;

    /**
     * Send messages
     *

     * @return array
     */
    public function send()
    {
        // Format 1 => MULTI MESSAGE ==> [[to: "+9779856034616", message: "This is test"], [to: "+9779856034617", message: "This is test"]]
        if (!empty($this->messages))
        {
            return $this->sendSmsToRecipient($this->messages);
        }

        // Format 3 => CAMPAIGN MESSAGE ==> to: ["+9779856034616", "+9779856034617"], message: "This is test"
        return $this->sendSmsToRecipient($this->prepareData($this->recipient, $this->message));
    }

    /**
     * Prepare common data
     *

     * @param array $recipients
     * @param string $message
     * @return array
     */
    public function prepareData(array $recipients, string $message)
    {
        $messages = [];
        foreach ($recipients as $phone) {
            $messages[] = ['phone' => $phone, 'text' => $message, '_id' => HashHelper::generateUniqueID()];
        }
        return $messages;
    }

    /**
     * Prepare and send messages
     *

     * @param array $messages
     * @return type
     */
    public function sendSmsToRecipient(array $messages)
    {
        $response = [];
        foreach ($messages as $message) {
            $this->getProviderData($message['provider']);
            $data['destinations'][] = [
              'to' => str_replace('+', '', $message['to']),
              'messageId' => $message['_id']
            ];
            $data['content'] = [
                'text' => $message['message']
            ];

            if (!empty($this->sender)) {
                $data['sender'] = $this->sender;
            }
            if ($message['provider']->callback_url) {
                $data['webhooks']['delivery']['url'] = $this->provider->callback_url;
                $data['webhooks']['contentType'] = "DLR callback data";
                $data['webhooks']['callbackData'] = "DLR callback data";
            }


            $providerResponse = $this->sendRequest($this->path, $this->provider->method, $data);
            if (!isset($message['_id']))
            {
                $response['FAILED'][] = $providerResponse;
            } else
            {
                $providerResponse['_id'] = $message['_id'];
                $response[$providerResponse['status']][] = $providerResponse;
            }
        }
        return $response;
    }

    /**
     * Get provider data
     *

     * @return array
     */
    public function getProviderData($provider)
    {
        $this->provider = $provider;

        $this->path = $this->provider->url;

        return [];
    }

    /**
     * Guzzle HTTP Request for all API Call
     *

     * @param       $path
     * @param       $method
     * @param array $data
     *
     * @return array
     */
    public function sendRequest($path, $method, $data = [])
    {
        try
        {
            $client = new Client([
                'headers' => [
                    'Content-Type' => 'application/json',
                    'Authorization' => "App 58d6c19bacc297071fbecef0555ec2dd-08d8f3f4-5696-4b50-921f-3e16689e4e6e"
                ]
            ]);
            $messages[] = $data;
            $resp = $client->post($path, ['json' => compact('messages')]);
            $response = json_decode($resp->getBody()->getContents(), true);

            if (isset($response[$this->getMessageField()]))
            {
                $messageId = $response[$this->getMessageField()];
                if (empty($messageId))
                {
                    $response = [
                        'error' => true,
                        'reason' => 'Cannot send to InfoBip Provider (empty message): ' . $resp,
                        'data' => $data,
                        'status' => 'REJECTED', // Initial rejection
                    ];
                } else
                {
                    $response = [
                        'message_id' => $messageId,
                        'status' => 'SENT-TO-PROVIDER',
                    ];
                }
            } elseif(isset($response['messages'][0][$this->getMessageField()])){

                $id0 = $response['messages'][0][$this->getMessageField()];
                $response = [
                    'message_id' => $id0,
                    'status' => 'SENT-TO-PROVIDER',
                ];
            }else
            {
                $response = [
                    'error' => true,
                    'reason' => 'Cannot send to InfoBip Provider (No phone in response): ' . $resp,
                    'data' => $data,
                    'status' => 'REJECTED',
                ];
            }
        } catch (\Exception $e)
        {
            $response = [
                'error' => true,
                'reason' => HashHelper::maskValue($data['password'] ?? '', $e->getMessage()),
                'data' => $data,
                'status' => 'REJECTED',
            ];
        }
        return $response;
    }

    /**
     * After sending message provider(provider have different fields in response), I want to map different responses to have standard mapping
     * With this function, I will be getting standardized fields for all providers
     *

     * @param array $resultMessage
     * @return array
     */
    public function mapper(array $resultMessage)
    {
        if (!isset($resultMessage[$this->getMessageField()], $resultMessage['status']['groupName']))
        {
            throw new CustomException('Invalid format', 400);
        }

        $resultMessage['provider'] = 'InfoBip';
        if (in_array(strtoupper($resultMessage['status']['groupName']), ['DELIVERED']))
        {
            $resultMessage['status'] = 'DELIVERED';
        }
        elseif ((in_array(strtoupper($resultMessage['status']['groupName']), ['PENDING'])))
        {
            $resultMessage['status'] = 'SENT';
        }
        elseif ((in_array(strtoupper($resultMessage['status']['groupName']), ['REJECTED', 'UNDELIVERED'])))
        {
            $resultMessage = $this->setFailedReasons($resultMessage);
        }
        elseif (isset($resultMessage[$this->getMessageField()]))
        {
            $resultMessage['status'] = 'FAILED';
        }
        elseif($resultMessage['status'] != "EXPIRED")
        {
            $resultMessage['status'] = 'REJECTED';
        }
        return $resultMessage;
    }

    /**
     * Receive callbacks from Provider
     *
     * @param $data
     * @return array
     * @throws CustomException
     */
    public function receiveCallback($data)
    {
        return $this->mapper($data['results'][0]);
    }

    /**
     * Set the priority
     *

     * @param array $requestData
     *
     * @return $this
     * @throws \Throwable
     */
    public function requestData($requestData)
    {
        $this->priority = $requestData['priority'];
        return parent::requestData($requestData);
    }

    /**
     * @return string
     */
    public function getMessageField()
    {
        return 'messageId';
    }

    /**
     * Set the "internal_failed_reason" and "failed_reason".
     *
     * @param array $resultMessage
     * @return array
     */
    private function setFailedReasons(array $resultMessage)
    {
        try {
            $resultMessage['internal_failed_reason'] = $resultMessage['status']['name'];
            $resultMessage['failed_reason'] = FailedReasonEnum::DEFAULT_CUSTOMER_FAILED_REASON;
        } catch (Throwable $exception) {
            app(GrayLogHandler::class)->setTag(LoggingTagEnum::MESSAGES_SERVICE)->error("InfoBipDriver@setFailedReasons: {$exception->getMessage()}", [
                'result_message_status' => $resultMessage['status'] ?? null,
                'result_message_id' => $resultMessage[$this->getMessageField()] ?? null
            ]);
            $resultMessage['internal_failed_reason'] = "Failed reason cannot be set because of the error in our code: {$exception->getMessage()}";
            $resultMessage['failed_reason'] = FailedReasonEnum::DEFAULT_CUSTOMER_FAILED_REASON;
        }

        return $resultMessage;
    }

    /**
     * Check if InfoBip DLR returns any error codes that match the ones on FailedReasonEnum constants
     * @param $status
     * @return bool
     */
    private function isStatusPartOfRejectedErrorCodes($status)
    {
        $error_types = FailedReasonEnum::getConstantValues();
        foreach($error_types as $error_type){
            if(strstr($status, $error_type) !== FALSE){
                return true;
            }
        }
        return false;
    }
}
