<?php

namespace Intergo\Providers\MessageManager\Drivers;

use Intergo\Log\GrayLogHandler;
use Intergo\Providers\Enums\LoggingTagEnum;
use Intergo\Providers\Enums\ViberCommonStatusEnum;
use Intergo\Providers\Exceptions\CustomException;
use GuzzleHttp\Client;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Cache;
use Intergo\Providers\Enums\FailedReasonEnum;
use Throwable;
use Intergo\Providers\Helpers\StringHelper;

class MstatViberDriver extends Driver
{
    const DELIVERED_STATUSES = [
        'delivered',
        'seen'
    ];
    const FAILED_STATUSES = [
        'expired',
    ];
    const REJECTED_STATUSES = [
        'insufficient_balance',
        'rejected',
        'destination_invalid',
        'destination_not_allowed'
    ];
    const REJECTED_ERROR_CODES = [
        ViberCommonStatusEnum::NOT_VIBER_USER,
        ViberCommonStatusEnum::NO_SUITABLE_DEVICE
    ];

    const UNSUBSCRIBED_ERROR_CODES = [
        ViberCommonStatusEnum::USER_BLOCKED,
    ];

    //CONSTS FOR VIBER ERRORS
    const ACCEPTED = "accepted";
    const DESTINATION_INVALID = "destination_invalid";

    //CONSTS FOR VIBER MESSAGE TYPES
    const ONE_WAY_TEXT = '106';
    const ONE_WAY_IMAGE = '107';
    const ONE_WAY_TEXT_BUTTON = '109';
    const ONE_WAY_TEXT_IMAGE_BUTTON = '108';
    const TWO_WAY_TEXT = '206';
    const TWO_WAY_IMAGE = '207';
    const TWO_WAY_TEXT_BUTTON = '209';
    const TWO_WAY_TEXT_IMAGE_BUTTON = '208';

    /**
     * Send messages
     *
     * @link https://intergo.atlassian.net/wiki/spaces/ES/pages/2377515026/M-STAT+API
     *
     * @return array
     */
    public function send()
    {
        $message = $this->messages[0];
        $params = $message['viber_params']['mstat'];

        $messageParams['text'] = $message['message'];

        $messageParams = array_merge($messageParams, !empty($message['viber_image_url']) ? ['image' => $message['viber_image_url'] ] : []);
        $messageParams = array_merge($messageParams, !empty($message['viber_caption']) ? ['caption' => $message['viber_caption'] ] : []);
        $messageParams = array_merge($messageParams, !empty($message['viber_target_url']) ? ['action' => $message['viber_target_url'] ] : []);

        $viberMessage = [
            'message' => $messageParams,
            'type' => intval($params['type']) ?? 208,
            'validity_period' => 14440,
        ];

        if (in_array(intval($params['type']), [
            self::TWO_WAY_TEXT,
            self::TWO_WAY_IMAGE,
            self::TWO_WAY_TEXT_BUTTON,
            self::TWO_WAY_TEXT_IMAGE_BUTTON,
        ])) {
            $viberMessage['tracking_data'] = $message['_id'];
        }

        $input = [
            'channel' => $params['channel'],
            'unique_id' => 'auto',
            'destination' => $message['to'],
            'dlr_url' => config('services.mstat.notify_url'),
            'flow' => [
                [
                    'viber' => $viberMessage,
                ]
            ]
        ];

        app(GrayLogHandler::class)->info('MSTAT Viber Request Payload', $input);

        return $this->sendRequest($input);
    }

    /**
     * Guzzle HTTP Request for all API Call
     *
     *
     * @param array $input
     *
     * @return array
     */
    public function sendRequest(array $input)
    {
        $message = $this->messages[0];
        $params = $message['viber_params']['mstat'];
        $client = new Client([
                                 'headers' => [
                                     'Content-Type' => 'application/json',
                                     'API-Token' => $params['api_token']
                                 ]
                             ]);
        try{
            $clientResponse = $client->post($params['url'], ['json' => [$input]]);
            $clientResponseArr = json_decode($clientResponse->getBody(), true);
            app(GrayLogHandler::class)->info('MSTAT Viber Response Payload', $clientResponseArr);
            $is_accepted = $clientResponseArr['data'][0]['status']['code'] == self::ACCEPTED;
            $response = array();
            if ($is_accepted) {
                $response['SENT-TO-PROVIDER'][] = [
                    '_id' => $message['_id'],
                    'message_id' => $clientResponseArr['data'][0][$this->getMessageField()] ?? null,
                    'status' =>'SENT-TO-PROVIDER',
                ];
            } else {
                $response['FAILED'][] = [
                    '_id' => $message['_id'],
                    'message_id' => $clientResponseArr['data'][0][$this->getMessageField()] ?? null,
                    'status' =>'FAILED',
                    'error' => true,
                    'reason' => $clientResponseArr['data'][0]['status']['description'],
                    'data' => json_encode($input),
                ];
            }

        } catch (\Exception $e) {
            $response['REJECTED'][] = [
                '_id' => $message['_id'],
                'status' =>'REJECTED',
                'error' => true,
                'reason' => $e->getMessage(),
                'data' => json_encode($input),
            ];
        }

        return $response;
    }

    /**
     * Set the "internal_failed_reason" and "failed_reason".
     *
     * @param array $resultMessage
     * @param string $status_details
     * @return array
     */
    protected function setFailedReasons(array $resultMessage, string $status_details)
    {
        try {
            $resultMessage['internal_failed_reason'] = $status_details;
            $resultMessage['failed_reason'] = FailedReasonEnum::getCustomerFailedReason($status_details);
        } catch (Throwable $exception) {
            app(GrayLogHandler::class)->setTag(LoggingTagEnum::MESSAGES_SERVICE)->error("MstatViberDriver@setFailedReasons: {$exception->getMessage()}", [
                'result_message_status' => $resultMessage['status'] ?? null,
                'result_message_id' => $resultMessage['campaignTrackingId'] ?? $resultMessage[$this->getMessageField()] ?? null
            ]);
            $resultMessage['internal_failed_reason'] = "Failed reason cannot be set because of the error in our code: {$exception->getMessage()}";
            $resultMessage['failed_reason'] = FailedReasonEnum::DEFAULT_CUSTOMER_FAILED_REASON;
        }

        return $resultMessage;
    }

    /**
     * After sending message provider(provider have different fields in response), I want to map different responses to have standard mapping
     * With this function, I will be getting standardized fields for all providers
     *
     *
     * @param array $resultMessage
     * @return array
     */
    public function mapper(array $resultMessage)
    {
        if (!isset($resultMessage[$this->getMessageField()])) {
            throw new CustomException('Invalid format', 400);
        }
        $status_code = $resultMessage['status']['code'] ?? "unknown";
        if (in_array($status_code, self::DELIVERED_STATUSES)) {
            $resultMessage['status'] = 'DELIVERED';
        } elseif (in_array($status_code, self::FAILED_STATUSES)) {
            $resultMessage['status'] = 'FAILED';
            $resultMessage = $this->setFailedReasons($resultMessage,$status_code);

        } elseif (in_array($status_code, self::REJECTED_STATUSES)) {
            $resultMessage['status'] = 'REJECTED';
            $resultMessage = $this->setFailedReasons($resultMessage,$status_code);
        } elseif (in_array(strtoupper($status_code), self::UNSUBSCRIBED_ERROR_CODES)){
            $resultMessage['status'] = 'UNSUBSCRIBED';
        } else {
            $resultMessage['status'] = 'FAILED';
            $resultMessage = $this->setFailedReasons($resultMessage,$status_code);
        }

        if (StringHelper::strpos_in_array(strtoupper($status_code), self::REJECTED_ERROR_CODES) !== false) {
            $resultMessage['status'] = 'REJECTED';
            $status_code = strtoupper($status_code);
            $resultMessage = $this->setFailedReasons($resultMessage, $status_code);
        }
        $resultMessage['message_id'] = $resultMessage[$this->getMessageField()];
        unset($resultMessage[$this->getMessageField()]);
        return $resultMessage;
    }

    /**
     * Receive callbacks from Provider
     *
     * @param $data
     * @return array
     * @throws CustomException
     */
    public function receiveCallback($data)
    {
        return $this->mapper($data);
    }

    /**
     * @return string
     */
    public function getMessageField()
    {
        return 'unique_id';
    }

}
