<?php

namespace Intergo\Providers\Enums;

use Intergo\Providers\MessageManager\Drivers\MstatViberDriver;
use Intergo\Providers\MessageManager\Drivers\NthViberDriver;

class FailedReasonEnum extends BaseEnum
{
    const DEFAULT_CUSTOMER_FAILED_REASON = 'Technical error 5008 - Contact Support';

    //ALARIS ERROR RESPONSES ON THE API EG. resulted in a `400 Bad Request` response: NO CLT RATE
    const CLT_AGREEMENT_EXPIRED = "CLT AGREEMENT EXPIRED";
    const NO_CLT_RATE = "NO CLT RATE";
    const NO_VDR_RATE = "NO VDR RATE";
    const NO_CLT_CUR = "NO CLT CUR";
    const NO_MCCMNC_FOR_DNIS = "NO MCCMNC FOR DNIS";
    const CLT_INBOUND_TRAFFIC_IS_PRHBT = "CLT INBOUND TRAFFIC IS PRHBT";
    const DISABLED_CHANNEL = "DISABLED CHANNEL";
    const CLT_PRDCT_NOT_FOUND = "CLT PRDCT NOT FOUND";
    const MSG_LMT_EXCEED_CLT_PRDCT_BLK = "MSG LMT EXCEED, CLT PRDCT BLK";
    const ROUTE_FOUND_FILTERED_BY_RULES = "ROUTE FOUND FILTERED BY RULES";
    const CLT_CARRIER_NOT_FOUND = "CLT CARRIER NOT FOUND";
    const CLT_ACC_NOT_FOUND = "CLT ACC NOT FOUND";
    const CLT_CL_EXCEEDED = "CLT CL EXCEEDED";
    const CLT_CHNL_OR_POI_NOT_FOUND = "CLT CHNL OR POI NOT FOUND";
    const CLT_POI_EXPIRED = "CLT POI EXPIRED";
    const ROUTE_FND_FLTRD_BY_VD_CHL_FLTR = "ROUTE FND FLTRD BY VD CHL FLTR";
    const NO_ROUTES_FOUND = "NO ROUTES FOUND";
    const ROUTE_FLTRD_OR_BLCKD = "ROUTE FLTRD OR BLCKD";
    const CLT_LOW_MPS_LIMIT = "CLT LOW MPS LIMIT";
    const CLT_PRDCT_LOCKED = "CLT PRDCT LOCKED";
    const IM_CHNL_INVALID = "IM CHNL INVALID";
    const IM_CHNL_REJECTED = "IM CHNL REJECTED";
    const UNKNOWN_ERROR = "unknown_error";

    //Alaris DLR responses that MUST be set to status REJECTED
    const CLN_LOW_BALANCE_MPS_LIMIT = "CLN LOW BALANCE MPS LIMIT";
    const CLN_PRODUCT_IS_LOCKED = "CLN PRODUCT IS LOCKED";
    const DATA_CODING_FAILED = "DATA CODING FAILED";
    const DATA_CODING_INCOMPATIBILITY = "DATA CODING INCOMPATIBILITY";
    const DNIS_MCCMNC_NOT_FND = "DNIS MCCMNC NOT FND";
    const HELD_SMS_NO_CHANNEL = "HELD SMS NO CHANNEL";
    const INCOMPLETE_LONG_MESSAGE = "INCOMPLETE LONG MESSAGE";
    const IM_CHANNELS_REJECTED = "IM CHANNELS REJECTED";
    const INCORRECT_MESSAGE_LENGTH_DC = "INCORRECT MESSAGE LENGTH (DC)";
    const MESSAGE_ENCODING_FAILED = "MESSAGE ENCODING FAILED";
    const NO_ACTIVE_SMSRT = "NO ACTIVE SMSRT";
    const NO_ROUTES = "NO ROUTES";
    const NO_TEST_VND_POI_FOUND = "NO TEST VND POI FOUND";
    const ROUTE_FAILED = "ROUTE FAILED";
    const ROUTE_DROPPED_BY_VENDOR_BLOCK = "ROUTE DROPPED BY VENDOR BLOCK";
    const ROUTE_VALIDITY_PERIOD_EXPIRED = "ROUTE VALIDITY PERIOD EXPIRED";
    const RULE_BUFFER_OVERFLOW = "RULE BUFFER OVERFLOW";
    const SMS_IN_BUF_OVL = "SMS IN BUF OVL";
    const SMS_IN_HOLD_BUF_OVL = "SMS IN HOLD BUF OVL";
    const SMS_IN_LIM_OVL = "SMS IN LIM OVL";
    const SMS_LOOP = "SMS LOOP";
    const SMS_OUT_BUF_OVL = "SMS OUT BUF OVL";
    const SMSRT_TIMEOUT = "SMSRT TIMEOUT";
    const SRC_POI_NOT_FOUND = "SRC POI NOT FOUND";
    const SUBMIT_RESP_TIMEOUT = "SUBMIT_RESP TIMEOUT";
    const VALIDITY_PERIOD_IS_EXPIRED = "VALIDITY PERIOD IS EXPIRED";
    const VALID_PERIOD_EXPIRED_BUFFER_RL = "VALID PERIOD EXPIRED-BUFFER RL";
    const VND_CHANNEL_UNKNOWN = "VND CHANNEL UNKNOWN";
    const VND_CHN_NOT_BND = "VND CHN NOT BND";
    const VND_CHN_NOT_HTTP_SMPP = "VND CHN NOT HTTP/SMPP";
    const VND_CHNL_SBM_QUEUE_OVL = "VND CHNL SBM QUEUE OVL";
    const VND_CHN_TCP_FAILED = "VND CHN TCP FAILED";
    const VND_CHN_TCP_TIMEOUT = "VND CHN TCP TIMEOUT";
    const WRONG_SEGMENT_NUMBER = "WRONG SEGMENT NUMBER";

    public static $failed_code_description = [
        self::CLN_LOW_BALANCE_MPS_LIMIT => "Client low balance MPS limit",
        self::CLN_PRODUCT_IS_LOCKED => "Client product is locked",
        self::CLT_ACC_NOT_FOUND =>"Client account not found",
        self::CLT_AGREEMENT_EXPIRED =>"Client agreement is outdated",
        self::CLT_CARRIER_NOT_FOUND =>"Client carrier not found",
        self::CLT_CHNL_OR_POI_NOT_FOUND =>"Client POI or channel not found",
        self::CLT_CL_EXCEEDED =>"Client credit limit exceeded",
        self::CLT_INBOUND_TRAFFIC_IS_PRHBT =>"Client inbound traffic is prohibited",
        self::CLT_LOW_MPS_LIMIT =>"Client low balance MPS limit",
        self::CLT_POI_EXPIRED =>"Client POI is outdated",
        self::CLT_PRDCT_LOCKED =>"Client product is locked",
        self::CLT_PRDCT_NOT_FOUND =>"Client product not found",
        self::DATA_CODING_FAILED => "Data Coding failed",
        self::DATA_CODING_INCOMPATIBILITY => "Data Coding incompatibility",
        self::DISABLED_CHANNEL =>"Client channel disabled",
        self::DNIS_MCCMNC_NOT_FND => "Destination number MCC MNC not found",
        self::HELD_SMS_NO_CHANNEL => "Held SMS no channel",
        self::IM_CHANNELS_REJECTED => "IM channel rejected",
        self::IM_CHNL_INVALID =>"IM channel invalid",
        self::IM_CHNL_REJECTED => "IM channels rejected",
        self::INCOMPLETE_LONG_MESSAGE => "Incomplete long message",
        self::INCORRECT_MESSAGE_LENGTH_DC => "Incorrect message length",
        self::MESSAGE_ENCODING_FAILED => "Message encoding failed",
        self::MSG_LMT_EXCEED_CLT_PRDCT_BLK =>"Message limit exceeded, client product blocked",
        self::NO_ACTIVE_SMSRT => "No active SMS Route",
        self::NO_CLT_CUR =>"Cannot find client account currency",
        self::NO_CLT_RATE =>"Client rate not found",
        self::NO_MCCMNC_FOR_DNIS =>"Cannot find MCCMNC for DNIS",
        self::NO_ROUTES => "No routes found",
        self::NO_ROUTES_FOUND =>"No routes found",
        self::NO_TEST_VND_POI_FOUND => "No test vendor POI found",
        self::NO_VDR_RATE =>"No vendor rates found",
        self::ROUTE_DROPPED_BY_VENDOR_BLOCK => "Route dropped by vendor block",
        self::ROUTE_FAILED => "Route failed",
        self::ROUTE_FLTRD_OR_BLCKD =>"Routes filtered out by blocking rules",
        self::ROUTE_FND_FLTRD_BY_VD_CHL_FLTR =>"Routes found but filtered by vendor channel filter",
        self::ROUTE_FOUND_FILTERED_BY_RULES =>"Routes found but filtered by rules",
        self::ROUTE_VALIDITY_PERIOD_EXPIRED => "Route validity period expired",
        self::RULE_BUFFER_OVERFLOW => "Rule buffer overflow",
        self::SMSRT_TIMEOUT => "SMS route timeout",
        self::SMS_IN_BUF_OVL => "SMS inbound buffer overflow",
        self::SMS_IN_HOLD_BUF_OVL => "SMS inbound hold buffer overflow",
        self::SMS_IN_LIM_OVL => "SMS inbound limit overflow",
        self::SMS_LOOP => "SMS loop",
        self::SMS_OUT_BUF_OVL => "SMS outbound buffer overflow",
        self::SRC_POI_NOT_FOUND => "Source point of interconnection not found",
        self::SUBMIT_RESP_TIMEOUT => "Submit response time out",
        self::VALIDITY_PERIOD_IS_EXPIRED => "Validity period expired",
        self::VALID_PERIOD_EXPIRED_BUFFER_RL => "Validity period expired buffer rule",
        self::VND_CHANNEL_UNKNOWN => "Vendor channel unknown",
        self::VND_CHNL_SBM_QUEUE_OVL => "Vendor channel submit queue overlfow",
        self::VND_CHN_NOT_BND => "Vendor channel not bound",
        self::VND_CHN_NOT_HTTP_SMPP => "Vendor channel not http/smpp",
        self::VND_CHN_TCP_FAILED => "Vendor channel can't connect to http (TCP request rejected)",
        self::VND_CHN_TCP_TIMEOUT => "Vendor channel TCP time out (no response to TCP request)",
        self::WRONG_SEGMENT_NUMBER => "Wrong segment number"
    ];

    /**
     * Get the error type so we can use it on Graylog or other places to group the rejected errors from providers
     * @param $reason
     * @return mixed|string
     */
    public static function getRejectedErrorType($reason)
    {
        $error_types = self::getConstantValues();
        foreach($error_types as $error_type){
            if(strstr($reason, $error_type) !== FALSE){
                return self::$failed_code_description[$error_type] ?? $error_type;
            }
        }
        return self::UNKNOWN_ERROR;
    }

    /**
     * Get the error type so we can use it on Graylog or other places to group the rejected errors from providers
     * @param $reason
     * @return mixed|string
     */
    public static function getViberFailedReason(string $viber_status_code)
    {
        switch ($viber_status_code) {
            case strstr($viber_status_code, ViberCommonStatusEnum::USER_BLOCKED) !== FALSE :
                return 'Technical error 5009 - Communication is blocked by recipient';
            case strstr($viber_status_code, ViberCommonStatusEnum::NOT_VIBER_USER) !== FALSE :
                return 'Technical error 5010 - User does not have Viber application installed on mobile device';
            case strstr($viber_status_code, ViberCommonStatusEnum::NO_SUITABLE_DEVICE) !== FALSE :
                return 'Technical error 5011 - Message not supported by mobile device';
            case strstr($viber_status_code, NthViberDriver::TIMEOUT) !== FALSE :
                return 'Technical error 5012 - Request timeout';
            case strstr($viber_status_code, NthViberDriver::INVALID_PHONE_NUMBER_FORMAT) !== FALSE :
            case strstr($viber_status_code, MstatViberDriver::DESTINATION_INVALID) !== FALSE :
                return 'Technical error 5013 - Invalid destination address (MSISDN), e.g. MSISDN not according to E.164 standard';
            case strstr($viber_status_code, NthViberDriver::REQUEST_ERROR) !== FALSE :
                return 'Technical error 5014 - Contact Support';
            case strstr($viber_status_code, NthViberDriver::INTERNAL_ERROR) !== FALSE :
                return 'Technical error 5015 - Contact Support';
        }
        return null;
    }

    /**
     * Get the 'failed_reason' that will be displayed to the customer.
     *
     * Here we map the error messages/statuses from providers with the error messages that will be visible to customers.
     *
     * @see https://app.asana.com/0/754800746265705/1201445985948718/f
     * @see https://intergo.atlassian.net/wiki/spaces/SMSD/pages/1660387332/User+Error+Codes
     * @param string $providerReason
     * @return string
     */
    public static function getCustomerFailedReason(string $providerReason)
    {

        $internal_reason = self::getViberFailedReason($providerReason);

        if ($internal_reason ) {
            return  $internal_reason;
        }

        switch ($providerReason) {
            // Routee specific: https://docs.routee.net/docs/statuses-callbacks
            case 'Undelivered on Handset':
                return 'Technical error 5001 - Undelivered on Handset';
            case 'Unknown Number':
                return 'Technical error 5002 - Unknown Number';
            case 'Roaming':
                return 'Technical error 5003 - Roaming';
            case 'Call Barred':
                return 'Technical error 5004 - Call Barred';
            case 'Unknown Status':
                return 'Technical error 5005 - Unknown Status';
            case 'Carrier Violation':
                return 'Technical error 5006 - Carrier Violation';
            case 'Expired':
                return 'Technical error 5007 - Expired';
            // If the error message/status from provider is not recognized/mapped:
            default:
                return self::DEFAULT_CUSTOMER_FAILED_REASON;
        }
    }
}
