<?php

namespace Intergo\Providers\MessageManager\Drivers;

use Intergo\Log\GrayLogHandler;
use Intergo\Providers\Enums\FailedReasonEnum;
use Intergo\Providers\Enums\LoggingTagEnum;
use Intergo\Providers\Exceptions\CustomException;
use Intergo\HashHelper\HashHelper;
use GuzzleHttp\Client;
use Throwable;

class AlarisDriver extends Driver
{
    protected $provider;

    protected $path;
    protected $backup_path;

    /**
     * Priority of the message
     *
     * @var int $priority
     */
    protected $priority = 2;

    /**
     * Send messages
     *

     * @return array
     */
    public function send()
    {
        // Format 1 => MULTI MESSAGE ==> [[to: "+9779856034616", message: "This is test"], [to: "+9779856034617", message: "This is test"]]
        if (!empty($this->messages))
        {
            return $this->sendSmsToRecipient($this->messages);
        }

        // Format 3 => CAMPAIGN MESSAGE ==> to: ["+9779856034616", "+9779856034617"], message: "This is test"
        return $this->sendSmsToRecipient($this->prepareData($this->recipient, $this->message));
    }

    /**
     * Prepare common data
     *

     * @param array $recipients
     * @param string $message
     * @return array
     */
    public function prepareData(array $recipients, string $message)
    {
        $messages = [];
        foreach ($recipients as $phone) {
            $messages[] = ['phone' => $phone, 'text' => $message, '_id' => HashHelper::generateUniqueID()];
        }
        return $messages;
    }

    /**
     * Prepare and send messages
     *

     * @param array $messages
     * @return type
     */
    public function sendSmsToRecipient(array $messages)
    {
        $response = [];
        foreach ($messages as $message) {
            $data = $this->getProviderData($message['provider']);
            $data['dnis'] = str_replace('+', '', $message['to']);
            $data['message'] = $message['message'];
            $data['flash'] = ($this->isFlashSMS) ? 1 : 0;

            if (!empty($this->sender)) {
                $data['ani'] = $this->sender;
            }
            $providerResponse = $this->sendRequest($data);
            if (!isset($message['_id'])) {
                $response['FAILED'][] = $providerResponse;
            } else {
                $providerResponse['_id'] = $message['_id'];
                $response[$providerResponse['status']][] = $providerResponse;
            }
        }
        return $response;
    }

    /**
     * Get provider data
     *

     * @return array
     */
    public function getProviderData($provider)
    {
        $this->provider = $provider;

        //http://62.67.222.162:8001/api?username=SMStoOTP&password=password&dnis=35799123456&message=Thisisatest&command=submit&longMessageMode=split_or_payload&flash=1
        $password = $this->provider->password;
        $username = $this->provider->username;
        $this->path = $this->provider->url;
        $this->backup_path = $this->provider->backup_url;

        $data = [
            'username' => $username,
            'password' => $password,
            'longMessageMode' => 'split_or_payload',
            'command' => 'submit'
        ];

        if ($this->provider->callback_url) {
            $data['dlr-url'] = $this->provider->callback_url;
        }

        return $data;
    }

    /**
     * Guzzle HTTP Request for all API Call
     *
     * @param array $data
     *
     * @return array
     */
    public function sendRequest($data = [])
    {
        $client = new Client();
        try {
            list($resp, $response) = $this->execute($client, $this->path, $data);
        } catch (Throwable $e) {
            app(GrayLogHandler::class)->error("Primary Url for $this->name failed. Attempting backup url", [
                'url' => $this->path,
                'response_message' => $e->getMessage(),
            ]);
            //retry on backup url
            try {
                list($resp, $response) = $this->execute($client, $this->backup_path, $data);
            } catch (Throwable $e) {
                app(GrayLogHandler::class)->error("Backup Url for $this->name failed", [
                    'url' => $this->backup_path,
                    'response_message' => $e->getMessage(),
                ]);
                return [
                    'error' => true,
                    'reason' => $e->getMessage(),
                    'data' => $data,
                    'status' => 'REJECTED',
                ];
            }
        }


        if (isset($response[$this->getMessageField()]))
        {
            $messageId = $response[$this->getMessageField()];
            if (empty($messageId))
            {
                $response = [
                    'error' => true,
                    'reason' => 'Cannot send to Alaris Provider (empty message): ' . $resp,
                    'data' => $data,
                    'status' => 'REJECTED', // Initial rejection
                ];
            } else
            {
                $response = [
                    'message_id' => $messageId,
                    'status' => 'SENT-TO-PROVIDER',
                ];
            }
        } elseif(isset($response[0][$this->getMessageField()])){

            $id0 = $response[0][$this->getMessageField()];
            $response = [
                'message_id' => $id0,
                'status' => 'SENT-TO-PROVIDER',
            ];
        }else
        {
            $response = [
                'error' => true,
                'reason' => 'Cannot send to Alaris Provider (No phone in response): ' . $resp,
                'data' => $data,
                'status' => 'REJECTED',
            ];
        }
        return $response;
    }

    /**
     * After sending message provider(provider have different fields in response), I want to map different responses to have standard mapping
     * With this function, I will be getting standardized fields for all providers
     *

     * @param array $resultMessage
     * @return array
     */
    public function mapper(array $resultMessage)
    {
        if (!isset($resultMessage[$this->getMessageField()], $resultMessage['state']))
        {
            throw new CustomException('Invalid format', 400);
        }

        $resultMessage['provider'] = 'Alaris';
        if (in_array(strtoupper($resultMessage['state']), ['DELIVRD', 'DELIVERED']))
        {
            $resultMessage['status'] = 'DELIVERED';
        } elseif ((in_array(strtoupper($resultMessage['state']), ['ESME_ROK', 'ENROUTE','ACCEPTD'])))
        {
            $resultMessage['status'] = 'SENT';
        } elseif (isset($resultMessage[$this->getMessageField()]) && $this->isStatusPartOfRejectedErrorCodes(strtoupper($resultMessage['state']))){
            $resultMessage['status'] = 'REJECTED';
            $resultMessage = $this->setFailedReasons($resultMessage,strtoupper($resultMessage['state']));
        }
        elseif (isset($resultMessage[$this->getMessageField()]))
        {
            $resultMessage['internal_failed_reason'] = $resultMessage['reason_code'] ?? '';
            $resultMessage['status'] = 'FAILED';
        }
        elseif($resultMessage['status'] != "EXPIRED")
        {
            $resultMessage['internal_failed_reason'] = $resultMessage['reason_code'] ?? '';
            $resultMessage['status'] = 'REJECTED';
        }
        return $resultMessage;
    }

    /**
     * Receive callbacks from Provider
     *
     * @param $data
     * @return array
     * @throws CustomException
     */
    public function receiveCallback($data)
    {
        return $this->mapper($data);
    }

    /**
     * Set the priority
     *

     * @param array $requestData
     *
     * @return $this
     * @throws \Throwable
     */
    public function requestData($requestData)
    {
        $this->priority = $requestData['priority'];
        return parent::requestData($requestData);
    }

    /**
     * @return string
     */
    public function getMessageField()
    {
        return 'message_id';
    }

    /**
     * Set the "internal_failed_reason" and "failed_reason".
     *
     * @param array $resultMessage
     * @param string $status_details
     * @return array
     */
    private function setFailedReasons(array $resultMessage, string $status_details)
    {
        try {
            $resultMessage['internal_failed_reason'] = $status_details;
            $resultMessage['failed_reason'] = FailedReasonEnum::DEFAULT_CUSTOMER_FAILED_REASON;
        } catch (Throwable $exception) {
            app(GrayLogHandler::class)->setTag(LoggingTagEnum::MESSAGES_SERVICE)->error("AlarisDriver@setFailedReasons: {$exception->getMessage()}", [
                'result_message_status' => $resultMessage['status'] ?? null,
                'result_message_id' => $resultMessage[$this->getMessageField()] ?? null
            ]);
            $resultMessage['internal_failed_reason'] = "Failed reason cannot be set because of the error in our code: {$exception->getMessage()}";
            $resultMessage['failed_reason'] = FailedReasonEnum::DEFAULT_CUSTOMER_FAILED_REASON;
        }

        return $resultMessage;
    }

    /**
     * Check if Alaris DLR returns any error codes that match the ones on FailedReasonEnum constants
     * @param $status
     * @return bool
     */
    private function isStatusPartOfRejectedErrorCodes($status)
    {
        $error_types = FailedReasonEnum::getConstantValues();
        foreach($error_types as $error_type){
            if(strstr($status, $error_type) !== FALSE){
                return true;
            }
        }
        return false;
    }

    /**
     * @param Client $client
     * @param $path
     * @param array $data
     * @return array
     */
    private function execute(Client $client, $path, array $data): array
    {
        $resp = $client->post($path, [
                                    'form_params'     => $data,
                                    'timeout'         => config('services.http.timeout', 30),   // max total request time (seconds)
                                    'connect_timeout' => config('services.http.connect_timeout', 15),   // optional: max time to connect (TCP)
                                    ])->getBody()->getContents();
        if (empty($resp)) {
            throw new \RuntimeException("Request cannot be completed. Timeout or empty response from provider.");
        }                                            
        $response = json_decode($resp, true);
        return [$resp, $response];
    }
}
