<?php


namespace Intergo\Providers\MessageManager\Drivers;

use Intergo\Providers\Exceptions\CustomException;
use GuzzleHttp\Client;
use Illuminate\Http\Request;
use Throwable;
use Intergo\Helpers\MessageHelper;

class JasminDriver extends Driver
{
    protected $provider;

    /**
     * @var
     */
    protected $path;

    /**
     * Priority of the message
     *
     * @var int $priority
     */
    protected $priority = 2;

    /**
     * {@inheritdoc}
     */
    public function send()
    {
        // Format 1 => MULTI MESSAGE ==> [[to: "+9779856034616", message: "This is test"], [to: "+9779856034617", message: "This is test"]]
        if (!empty($this->messages))
        {
            return $this->sendSmsToRecipient($this->messages);
        }

        // Format 3 => CAMPAIGN MESSAGE ==> to: ["+9779856034616", "+9779856034617"], message: "This is test"
        return $this->sendSmsToRecipient($this->prepareData($this->recipient, $this->message));

    }

    public function prepareData(array $recipients, string $message)
    {
        $messages = [];
        foreach ($recipients as $phone)
        {
            $messages[] = ['to' => $phone, 'message' => $message, '_id' =>generateUuid()];
        }
        return $messages;
    }

    public function sendSmsToRecipient(array $messages)
    {
        $response = [];
        foreach ($messages as $message)
        {
            $data = $this->getProviderData();
            $data['to'] = str_replace('+', '', $message['to']);
            $strEncode = mb_detect_encoding($message['message']);
            $data['content'] = mb_convert_encoding($message['message'], 'UTF-8', $strEncode);
            if (MessageHelper::isGsm7bit($message['message']))
            {
                $message['message'] = mb_convert_encoding($message['message'], 'UTF-8', $strEncode);
                $text = bin2hex(mb_convert_encoding($message['message'], 'UTF-16', 'UTF-8'));
                $data['hex-content'] = $text;
                unset($data['content']);
                $data['coding'] = 8;
            }

            if (!empty($this->sender))
            {
                $data['from'] = $this->sender;
            }
            $data['priority'] = $this->priority;
            $providerResponse = $this->sendRequest($this->path, 'POST', $data);
            if(!isset($message['_id']))
            {
                $response['FAILED'][] = $providerResponse;
            }
            else
            {
                $providerResponse['_id'] = $message['_id'];
                $response[$providerResponse['status']][] = $providerResponse;
            }

        }

        return $response;
    }

    /**
     * @param int $coding
     *
     * @return array
     */
    public function getProviderData()
    {
        $this->provider = $this->config;
        //http://116.203.125.176:1401/send?username=MTN&password=123456&from=SMSTO&to=35799585001&content=THIS%20IS%20A%20TEST%20SMS&coding=0&priority=0&dlr=yes&dlr-url=https%3A//sms.to/callback/jsm&dlr-level=3&dlr-method=POST

        $ip = $this->provider->ip;
        $port = $this->provider->port;
        $username = $this->provider->username;
        $password = $this->provider->password;
        $this->path = "http://$ip:$port/send";
        if(!empty($this->provider->url))
        {
            $this->path = $this->provider->url;
        }

        $data = [
            'username' => $username,
            'password' => $password,
            'coding' => 0,
        ];
        if ($this->provider->callback_url)
        {
            $data['dlr'] = 'yes';
            $data['dlr-url'] = $this->provider->callback_url;
            $data['dlr-level'] = $this->provider->report_level;
            $data['dlr-method'] = $this->provider->method;
        }
        return $data;
    }

    /**
     * Guzzle HTTP Request for all API Call
     *
     * @param       $path
     * @param       $method
     * @param array $data
     *
     * @return array
     * @author Sujit Baniya <sujit@intergo.com.cy>
     */
    public function sendRequest($path, $method, $data = [])
    {
        $client = new Client();
        try
        {
            $resp = $client->post($path, [
                'form_params' => $data,
            ])->getBody()->getContents();

            if (strpos($resp, 'Success') !== false)
            {
                $messageId = \Ramsey\Uuid\Uuid::fromString(str_replace(['Success ', '"'], '', $resp))->toString();
                if(empty($messageId))
                {
                    $response = [
                        'error' => true,
                        'reason' => 'Cannot send to Jasmin Provider: ' . $resp,
                        'data' => $data,
                        'status' => 'REJECTED', // Initial rejection
                    ];
                }
                else
                {
                    $response = [
                        'message_id' => $messageId,
                        'status' => 'SENT-TO-PROVIDER',
                    ];
                }
            } else
            {
                $response = [
                    'error' => true,
                    'reason' => 'Cannot send to Jasmin Provider: ' . $resp,
                    'data' => $data,
                    'status' => 'REJECTED',
                ];
            }
        }
        catch (\Exception $e)
        {
            $response = [
                'error' => true,
                'reason' => $e->getMessage(),
                'data' => $data,
                'status' => 'REJECTED',
            ];
        }
        return $response;
    }

    public function prepareResponse($response)
    {

    }

    /**
     * After sending message provider(provider have different fields in response), I want to map different responses to have standard mapping
     * With this function, I will be getting standardized fields for all providers
     *
     * {@inheritdoc}
     */
    public function mapper(array $resultMessage)
    {

        if(!isset($resultMessage[$this->getMessageField()], $resultMessage['message_status']))
        {
            throw new CustomException('Invalid format', 400);
        }

        $resultMessage['message_id'] = $resultMessage[$this->getMessageField()];
        $resultMessage['provider'] = 'Jasmin';
        if(in_array(strtoupper($resultMessage['message_status']), ['DELIVRD', 'DELIVERED']))
        {
            $resultMessage['status'] = 'DELIVERED';
        }
        elseif(strtoupper($resultMessage['message_status']) === 'ESME_ROK')
        {
            $resultMessage['status'] = 'SENT';
        }
        else
        {
            $resultMessage['status'] = 'FAILED';
        }
        return $resultMessage;
    }

    /**
     * Receive callbacks from Provider
     *
     * @param $data
     * @return array
     * @throws CustomException
     */
    public function receiveCallback($data)
    {
        return $this->mapper($data);
    }

    /**
     * Set the priority
     *
     * @author Panayiotis Halouvas <phalouvas@kainotomo.com>
     *
     * @param array $requestData
     *
     * @return $this
     * @throws \Throwable
     */
    public function requestData($requestData)
    {
        $this->priority = $requestData['priority'];
        return parent::requestData($requestData);
    }

    /**
     * @return string
     */
    public function getMessageField()
    {
        return 'id';
    }
}
