<?php

namespace Intergo\Providers\MessageManager\Drivers;

use Illuminate\Support\Facades\Redis;
use Intergo\Providers\Exceptions\CustomException;
use Intergo\Helpers\HashHelper;
use GuzzleHttp\Client;
use Illuminate\Http\Request;

class ItsmppDriver extends Driver
{
    protected $provider;

    /**
     * @var
     */
    protected $path;

    /**
     * Priority of the message
     *
     * @var int $priority
     */
    protected $priority = 2;

    /**
     * Send messages
     *
     * @author Panayiotis Halouvas <phalouvas@kainotomo.com>
     *
     * @return array
     */
    public function send()
    {
        // Format 1 => MULTI MESSAGE ==> [[to: "+9779856034616", message: "This is test"], [to: "+9779856034617", message: "This is test"]]
        if (!empty($this->messages))
        {
            return $this->sendSmsToRecipient($this->messages);
        }

        // Format 3 => CAMPAIGN MESSAGE ==> to: ["+9779856034616", "+9779856034617"], message: "This is test"
        return $this->sendSmsToRecipient($this->prepareData($this->recipient, $this->message));
    }

    /**
     * Prepare common data
     *
     * @author Panayiotis Halouvas <phalouvas@kainotomo.com>
     *
     * @param array $recipients
     * @param string $message
     * @return array
     */
    public function prepareData(array $recipients, string $message)
    {
        $messages = [];
        foreach ($recipients as $phone) {
            $messages[] = ['phone' => $phone, 'text' => $message, '_id' => HashHelper::generateUniqueID()];
        }
        return $messages;
    }

    /**
     * Prepare and send messages
     *
     * @author Panayiotis Halouvas <phalouvas@kainotomo.com>
     *
     * @param array $messages
     * @return type
     */
    public function sendSmsToRecipient(array $messages)
    {
        $response = [];
        foreach ($messages as $message) {
            $data = $this->getProviderData($message['provider']);
            $data['phone'] = str_replace('+', '', $message['to']);
            $data['text'] = $message['message'];

            if (!empty($this->sender))
            {
                $data['senderID'] = $this->sender;
            }
            $providerResponse = $this->sendRequest($this->path, 'GET', $data);
            if (!isset($message['_id']))
            {
                $response['FAILED'][] = $providerResponse;
            } else
            {
                $providerResponse['_id'] = $message['_id'];
                $response[$providerResponse['status']][] = $providerResponse;
            }
        }
        return $response;
    }

    /**
     * Get provider data
     *
     * @author Panayiotis Halouvas <phalouvas@kainotomo.com>
     *
     * @return array
     */
    public function getProviderData($provider)
    {
        $this->provider = $provider;
        //https://api.intergotelecom.com/send?token=mSkYudOab47F2ySd4NpobviqEqFdNH&phone=+357123456&senderID=SmsTo&text=Hello%20world
        $ip = $this->provider->ip;
        $port = $this->provider->port;
        $password = $this->provider->password;
        $this->path = "http://$ip:$port/send";
        if (!empty($this->provider->url))
        {
            $this->path = $this->provider->url;
        }

        $data = [
            'token' => $password,
        ];
        if ($this->provider->callback_url)
        {
            $data['dlr'] = 'yes';
            $data['dlr-url'] = $this->provider->callback_url;
            $data['dlr-level'] = $this->provider->report_level;
            $data['dlr-method'] = $this->provider->method;
        }
        return $data;
    }

    /**
     * Guzzle HTTP Request for all API Call
     *
     * @author Panayiotis Halouvas <phalouvas@kainotomo.com>
     *
     * @param       $path
     * @param       $method
     * @param array $data
     *
     * @return array
     */
    public function sendRequest($path, $method, $data = [])
    {
        $client = new Client();
        $ttl_itsmpp = 3600 * 24 * 3; //3 days
        try
        {
            $resp = $client->get($path, ['query' => $data])->getBody()->getContents();
            $response = json_decode($resp, true);
            if (isset($response[$data['phone']]))
            {
                $messageId = $response[$data['phone']];
                if (empty($messageId))
                {
                    $response = [
                        'error' => true,
                        'reason' => 'Cannot send to Itsmpp Provider (empty message): ' . $resp,
                        'data' => $data,
                        'status' => 'REJECTED', // Initial rejection
                    ];
                } elseif (is_array($messageId)) {
                    $id0 = $messageId[0][$this->getMessageField()];
                    $response = [
                        'message_id' => $id0,
                        'status' => 'SENT-TO-PROVIDER',
                    ];
                } else
                {
                    $response = [
                        'message_id' => $messageId,
                        'status' => 'SENT-TO-PROVIDER',
                    ];
                }
            } else
            {
                $response = [
                    'error' => true,
                    'reason' => 'Cannot send to Itsmpp Provider (No phone in response): ' . $resp,
                    'data' => $data,
                    'status' => 'REJECTED',
                ];
            }
        } catch (\Exception $e)
        {
            $response = [
                'error' => true,
                'reason' => HashHelper::maskValue($data['token'] ?? '', $e->getMessage()),
                'data' => $data,
                'status' => 'REJECTED',
            ];
        }
        return $response;
    }

    /**
     * After sending message provider(provider have different fields in response), I want to map different responses to have standard mapping
     * With this function, I will be getting standardized fields for all providers
     *
     * @author Panayiotis Halouvas <phalouvas@kainotomo.com>
     *
     * @param array $resultMessage
     * @return array
     */
    public function mapper(array $resultMessage)
    {
        if (!isset($resultMessage[$this->getMessageField()], $resultMessage['state']))
        {
            throw new CustomException('Invalid format', 400);
        }

        $resultMessage['message_id'] = $resultMessage[$this->getMessageField()];
        $resultMessage['provider'] = 'Itsmpp';
        if (in_array(strtoupper($resultMessage['state']), ['DELIVRD', 'DELIVERED']))
        {
            $resultMessage['status'] = 'DELIVERED';
        } elseif (strtoupper($resultMessage['state']) === 'ESME_ROK')
        {
            $resultMessage['status'] = 'SENT';
        }
        elseif (isset($resultMessage[$this->getMessageField()]))
        {
            $resultMessage['status'] = 'FAILED';
        }
        else
        {
            $resultMessage['status'] = 'REJECTED';
        }
        return $resultMessage;
    }

    /**
     * Receive callbacks from Provider
     *
     * @param $data
     * @return array
     * @throws CustomException
     */
    public function receiveCallback($data)
    {
        return $this->mapper($data);
    }

    /**
     * Set the priority
     *
     * @author Panayiotis Halouvas <phalouvas@kainotomo.com>
     *
     * @param array $requestData
     *
     * @return $this
     * @throws \Throwable
     */
    public function requestData($requestData)
    {
        $this->priority = $requestData['priority'];
        return parent::requestData($requestData);
    }

    /**
     * @return string
     */
    public function getMessageField()
    {
        return 'id_state';
    }
}
