<?php

namespace Intergo\ProductsRatesClient;

use Intergo\BaseClient\Responses\BaseResponseInterface;
use phpseclib3\Math\PrimeField\Integer;

abstract class AbstractProductsRatesService
{
    protected ProductsRatesClient $client;

    /**
     * AbstractMessageService constructor.
     *
     * @param $baseUrl
     * @param $traceId
     */
    public function __construct($baseUrl, $traceId)
    {
        $this->client = new ProductsRatesClient($baseUrl, $traceId);
    }

    /**
     * @param string $endpoint
     * @param array $urlParams
     * @param array $bodyParams
     * @return BaseResponseInterface
     */
    protected function execute(string $endpoint, array $urlParams = [], array $bodyParams = []): BaseResponseInterface
    {
        return $this->response($endpoint, $this->client->execute($endpoint, $urlParams, $bodyParams));
    }

    /**
     * @param string $action
     * @param BaseResponseInterface $result
     * @return BaseResponseInterface
     */
    abstract public function response(string $action, BaseResponseInterface $result): BaseResponseInterface;


    /**
     * Retrieves a list of networks with pagination
     *
     * @param string|null $mccMnc The mccMnc of the networks to be retrieved.
     * @param string|null $countryDialCode The country dial code of the networks to be retrieved.
     * @param string|null $name The name of the networks to be retrieved.
     * @param string|null $countryName The country name of the networks to be retrieved.
     * @param string|null $externalId The external id of the networks to be retrieved.
     * @param int|null $page The page of networks to be retrieved.
     * @param int|null $pageSize The page size of networks to be retrieved.
     * @param string|null $orderBy The order by property of the network to be retrieved.
     * @param string|null $direction The order by directions of the network to be retrieved.
     * @return BaseResponseInterface
     */
    public function listNetworks(
        ?string $mccMnc = null,
        ?string $countryDialCode = null,
        ?string $name = null,
        ?string $countryName = null,
        ?string $externalId = null,
        ?int $page = null,
        ?int $pageSize = null,
        ?string $orderBy = null,
        ?string $direction = null
    ) : BaseResponseInterface {
        $urlParams = [
            'mcc_mnc' => $mccMnc,
            'country_dial_code' => $countryDialCode,
            'name' => $name,
            'country_name' => $countryName,
            'external_id' => $externalId,
            'page' => $page,
            'page_size' => $pageSize,
            'order_by' => $orderBy,
            'direction' => $direction,
        ];

        return $this->execute(__FUNCTION__,[], $urlParams );
    }

    /**
     * Retrieves network by id.
     * @param int $id The ID of the network to be retrieved.
     * @return BaseResponseInterface
     */
    public function getNetworkById(int $id): BaseResponseInterface
    {
        return $this->execute(__FUNCTION__, compact('id'), []);
    }

    /**
     * Retrieves the network of a phone number.
     * @param string $phoneNumber The phone number that the network should be extracted.
     * @return BaseResponseInterface
     */
    public function getNetworkByPhoneNumber(string $phoneNumber): BaseResponseInterface
    {
        return $this->execute(__FUNCTION__, ['phone_number' => $phoneNumber], []);
    }

    /**
     * Retrieves a list of product adjusted rates with pagination
     *
     * @param string|null $userId The user id of the adjusted rates to be retrieved.
     * @param string|null $productId The product id of the adjusted rates to be retrieved.
     * @param string|null $mcc The mcc of the adjusted rates to be retrieved.
     * @param string|null $mccMnc The mccMnc of the adjusted rates to be retrieved.
     * @param string|null $senderId The sender id of the adjusted rates to be retrieved.
     * @param int|null $page The page of networks to be retrieved.
     * @param int|null $pageSize The page size of networks to be retrieved.
     * @param string|null $orderBy The order by property of the network to be retrieved.
     * @param string|null $direction The order by directions of the network to be retrieved.
     * @return BaseResponseInterface
     */
    public function listAdjustedRates(
        ?string $userId = null,
        ?string $productId = null,
        ?string $mcc = null,
        ?string $mccMnc = null,
        ?string $senderId = null,
        ?int $page = null,
        ?int $pageSize = null,
        ?string $orderBy = null,
        ?string $direction = null
    ) : BaseResponseInterface {
        $urlParams = [
            'user_id' => $userId,
            'product_id' => $productId,
            'mcc' => $mcc,
            'mnc_mnc' => $mccMnc,
            'sender_id' => $senderId,
            'page' => $page,
            'page_size' => $pageSize,
            'order_by' => $orderBy,
            'direction' => $direction,
        ];

        return $this->execute(__FUNCTION__,[], $urlParams );
    }

    /**
     * Retrieves an adjusted rate by id.
     * @param int $id The ID of the adjusted rate to be retrieved.
     * @return BaseResponseInterface
     */
    public function getAdjustedRateById(int $id): BaseResponseInterface
    {
        return $this->execute(__FUNCTION__, compact('id'), []);
    }

    /**
     * Creates an adjusted rate
     * @param array $data
     * @return BaseResponseInterface
     */
    public function createAdjustedRate(array $data) : BaseResponseInterface
    {
        return $this->execute(__FUNCTION__, [], $data);
    }

    /**
     * Updates an adjusted rate by id
     * @param int $id The ID of the adjusted rate to be updated.
     * @param array $data
     * @return BaseResponseInterface
     */
    public function updateAdjustedRate(int $id, array $data) : BaseResponseInterface
    {
        return $this->execute(__FUNCTION__, compact('id'), $data);
    }

    /**
     * Deletes an adjusted rate by id
     * @param int $id The ID of the adjusted rate to be deleted.
     * @return BaseResponseInterface
     */
    public function deleteAdjustedRate(int $id) : BaseResponseInterface
    {
        return $this->execute(__FUNCTION__, compact('id'));
    }

    /**
     * Retrieves a list of products with pagination
     *
     * @param string|null $caption The caption of the product to be retrieved.
     * @param string|null $description The description of the product to be retrieved.
     * @param string|null $externalId The external id  of the product to be retrieved.
     * @param string|null $typeIntegration The Product Type of the product to be retrieved.
     * @param int|null $page The page of product to be retrieved.
     * @param int|null $pageSize The page size of product to be retrieved.
     * @param string|null $orderBy The order by property of the product to be retrieved.
     * @param string|null $direction The order by directions of the product to be retrieved.
     * @param string|null $billingType The billing type of the product to be retrieved
     * @return BaseResponseInterface
     */
    public function listProducts(
        ?string $name = null,
        ?string $rateName = null,
        ?string $externalId = null,
        ?string $typeId = null,
        ?string $typeIntegration = null,
        ?string $direction = null,
        ?string $billingType = null,
        ?string $status = null,
        ?int $page = null,
        ?int $pageSize = null,
        ?string $orderBy = null
    ) : BaseResponseInterface {
        $urlParams = [
            'name' => $name,
            'rate_name' => $rateName,
            'external_id' => $externalId,
            'type_id' => $typeId,
            'type_integration' => $typeIntegration,
            'product_direction' => $direction,
            'billing_type'  => $billingType ,
            'status'  => $status ,
            'page' => $page,
            'page_size' => $pageSize,
            'order_by' => $orderBy,
        ];

        return $this->execute(__FUNCTION__,[], $urlParams );
    }

    /**
     * Retrieves product by id.
     * @param int $id The ID of the product to be retrieved.
     * @return BaseResponseInterface
     */
    public function getProductById(int $id): BaseResponseInterface
    {
        return $this->execute(__FUNCTION__, compact('id'), []);
    }

    /**
     * Updates a product by id
     * @param int $id The ID of the product to be updated.
     * @param array $data
     * @return BaseResponseInterface
     */
    public function updateProduct(int $id, array $data) : BaseResponseInterface
    {
        return $this->execute(__FUNCTION__, compact('id'), $data);
    }

    public function getProductRates(
        ?string $id= null,
        ?int $page = null,
        ?int $pageSize = null,
        ?string $orderBy = null,
        ?string $direction = null,
    ) : BaseResponseInterface {
        $urlParams = [
            'page' => $page,
            'page_size' => $pageSize,
            'order_by' => $orderBy,
            'direction' => $direction,
        ];

        return $this->execute(__FUNCTION__,compact('id'), $urlParams );
    }

    /**
     * Retrieves the rate of a specific phone number and user
     *
     * @param string $userId The user id the rate should be calculated for.
     * @param string|null $productId The product id the rate should be calculated for.
     * @param string $phoneNumber The phone number the rate should be calculated for.
     * @param string|null $senderId The sender id the rate should be calculated for.
     * @return BaseResponseInterface
     */
    public function getRate(
        string $userId,
        string $phoneNumber,
        ?string $productId = null,
        ?string $senderId = null
    ) : BaseResponseInterface {
        $urlParams = [
            'user_id' => $userId,
            'product_id' => $productId,
            'phone_number' => $phoneNumber,
            'sender_id' => $senderId,
        ];

        return $this->execute(__FUNCTION__,[], $urlParams );
    }

    /**
     * Retrieves a list of user products with pagination
     *
     * @param string|null $userId The user id of the user products to be retrieved.
     * @param string|null $productId The product id of the user products to be retrieved.
     * @param string|null $isDefault The default user products to be retrieved.
     * @param int|null $page The page of networks to be retrieved.
     * @param int|null $pageSize The page size of networks to be retrieved.
     * @param string|null $orderBy The order by property of the network to be retrieved.
     * @param string|null $direction The order by directions of the network to be retrieved.
     * @return BaseResponseInterface
     */
    public function listUserProducts(
        ?string $userId = null,
        ?string $productId = null,
        ?string $isDefault = null,
        ?int $page = null,
        ?int $pageSize = null,
        ?string $orderBy = null,
        ?string $direction = null
    ) : BaseResponseInterface {
        $urlParams = [
            'user_id' => $userId,
            'product_id' => $productId,
            'is_default' => $isDefault,
            'page' => $page,
            'page_size' => $pageSize,
            'order_by' => $orderBy,
            'direction' => $direction,
        ];

        return $this->execute(__FUNCTION__,[], $urlParams );
    }

    /**
     * Retrieves a user product by id.
     * @param int $id The ID of the user product to be retrieved.
     * @return BaseResponseInterface
     */
    public function getUserProductById(int $id): BaseResponseInterface
    {
        return $this->execute(__FUNCTION__, compact('id'), []);
    }

    /**
     * Creates a user product
     * @param array $data
     * @return BaseResponseInterface
     */
    public function createUserProduct(array $data) : BaseResponseInterface
    {
        return $this->execute(__FUNCTION__, [], $data);
    }

    /**
     * Deletes a user product by id
     * @param int $id The ID of the user product to be deleted.
     * @return BaseResponseInterface
     */
    public function deleteUserProduct(int $id) : BaseResponseInterface
    {
        return $this->execute(__FUNCTION__, compact('id'));
    }


    /**
     * Creates a user product
     * @param array $data
     * @return BaseResponseInterface
     */
    public function createUserProductAndAdjustedRatesForABrand(array $data) : BaseResponseInterface
    {
        return $this->execute(__FUNCTION__, [], $data);
    }

    /**
     * Retrieves a list of product types with pagination
     *
     * @param string|null $name The name of the product types to be retrieved.
     * @param string|null $integration The integration of the product types to be retrieved.
     * @param int|null $isDefault Retrieve the default product type.
     * @param int|null $page The page of networks to be retrieved.
     * @param int|null $pageSize The page size of networks to be retrieved.
     * @param string|null $orderBy The order by property of the network to be retrieved.
     * @param string|null $direction The order by directions of the network to be retrieved.
     * @return BaseResponseInterface
     */
    public function listProductTypes(
        ?string $name = null,
        ?string $integration = null,
        ?int $isDefault = null,
        ?int $page = null,
        ?int $pageSize = null,
        ?string $orderBy = null,
        ?string $direction = null
    ) : BaseResponseInterface {
        $urlParams = [
            'name' => $name,
            'integration' => $integration,
            'is_default' => $isDefault,
            'page' => $page,
            'page_size' => $pageSize,
            'order_by' => $orderBy,
            'direction' => $direction,
        ];

        return $this->execute(__FUNCTION__,[], $urlParams );
    }

    /**
     * Retrieves a product type by id.
     * @param int $id The ID of the product type to be retrieved.
     * @return BaseResponseInterface
     */
    public function getProductTypeById(int $id): BaseResponseInterface
    {
        return $this->execute(__FUNCTION__, compact('id'), []);
    }

    /**
     * Creates a product type
     * @param array $data
     * @return BaseResponseInterface
     */
    public function createProductType(array $data) : BaseResponseInterface
    {
        return $this->execute(__FUNCTION__, [], $data);
    }

    /**
     * Updates a product type by id
     * @param int $id The ID of the product type to be updated.
     * @param array $data
     * @return BaseResponseInterface
     */
    public function updateProductType(int $id, array $data) : BaseResponseInterface
    {
        return $this->execute(__FUNCTION__, compact('id'), $data);
    }

    /**
     * Deletes a product type by id
     * @param int $id The ID of the product type to be deleted.
     * @return BaseResponseInterface
     */
    public function deleteProductType(int $id) : BaseResponseInterface
    {
        return $this->execute(__FUNCTION__, compact('id'));
    }
}
