<?php

namespace Intergo\PhoneValidatorClient;

use Intergo\BaseClient\Responses\BaseResponseInterface;
use Intergo\PhoneValidatorClient\Exceptions\PhoneHelperInvalidPhoneNumberException;
use Intergo\PhoneValidatorClient\ValueObjects\PhoneNumberVO;

abstract class AbstractPhoneValidatorService
{
    protected PhoneValidatorClient $client;

    /**
     * @param $baseUrl
     * @param $traceId
     */
    public function __construct($baseUrl, $traceId)
    {
        $this->client = new PhoneValidatorClient($baseUrl, $traceId);
    }

    /**
     * @param $endpoint
     * @param $bodyParams
     * @param array $urlParams
     * @return BaseResponseInterface
     */
    protected function execute($endpoint, $bodyParams, array $urlParams = [])
    {
        $result = $this->client->execute($endpoint, $urlParams, $bodyParams);
        return $this->response($endpoint, $result);
    }

    /**
     * @param $endpoint
     * @param $token
     * @param $bodyParams
     * @param array $urlParams
     * @return BaseResponseInterface
     */
    protected function executeAuth($endpoint, $token, $bodyParams, array $urlParams = [])
    {
        $result = $this->client->executeAuth($endpoint, $token, $urlParams, $bodyParams);
        return $this->response($endpoint, $result);
    }

    /**
     * @param string $action
     * @param BaseResponseInterface $result
     * @return BaseResponseInterface
     */
    abstract public function response(string $action, BaseResponseInterface $result);

    /**
     * @param ?string $phoneNumber
     * @param ?string $defaultPrefix
     * @param bool $returnBool
     * @return bool|PhoneNumberVO
     * @throws PhoneHelperInvalidPhoneNumberException
     */
    public function validatePhoneNumber(?string $phoneNumber, string $defaultPrefix = null, bool $returnBool = false)
    {
        if (is_null($phoneNumber)) {
            throw new PhoneHelperInvalidPhoneNumberException('Empty phone number');
        }
        $result = $this->validatePhoneByRegion($phoneNumber, $defaultPrefix);
        if ($returnBool) {
            return boolval($result->isSuccessful());
        }
        if (!$result->isSuccessful()) {
            throw new PhoneHelperInvalidPhoneNumberException($result->getMessage());
        }
        $data = $result->getData();
        $phoneNumberVO = new PhoneNumberVO();
        $phoneNumberVO->setCountryCode($data['country_code']);
        $phoneNumberVO->setNetworkName($data['network_name']);
        $phoneNumberVO->setPhoneNumber($data['phone_number_as_string']);
        $phoneNumberVO->setNationalNumber($data['phone_number']['national_number']);
        $phoneNumberVO->setA2Code($data['a2_code']);
        $phoneNumberVO->setPhoneNumberType($data['phone_number_type']);
        return $phoneNumberVO;
    }

    /**
     * @param $phone
     * @return BaseResponseInterface
     */
    protected function validatePhone($phone)
    {
        return $this->execute(__FUNCTION__, [], compact('phone'));
    }

    /**
     * @param $phone
     * @param $region
     * @return BaseResponseInterface
     */
    protected function validatePhoneByRegion($phone, $region)
    {
        return $this->execute(__FUNCTION__, [], compact('phone', 'region'));
    }
}