<?php

namespace Intergo\OptOut\Repositories;

use Intergo\OptOut\Enums\ContactsEnum;
use Illuminate\Support\Collection;
use Illuminate\Support\Facades\DB;

class ContactRepository
{
    protected DB $db;

    public function __construct(DB $db)
    {
        $this->db = $db;
    }

    /**
     * @param $appUserId
     * @return Collection
     */
    public function getOptOutsByUserId($appUserId)
    {
        return $this->db::connection(ContactsEnum::DB_CONNECTION)->table(ContactsEnum::CONTACTS_TABLE)
            ->where('is_optout', 1)
            ->where('user_id', $appUserId)
            ->get('phone');
    }

    /**
     * @param array $listIds
     * @return Collection
     */
    public function getOptOutsByListIds(array $listIds)
    {
        return $this->db::connection(ContactsEnum::DB_CONNECTION)->table(ContactsEnum::CONTACTS_TABLE)
            ->leftJoin(ContactsEnum::LIST_CONTACTS_TABLE,
                ContactsEnum::CONTACTS_TABLE . '.id', '=', ContactsEnum::LIST_CONTACTS_TABLE . '.contact_id')
            ->leftJoin(ContactsEnum::USER_LISTS_TABLE, function ($join) {
                $join->on(ContactsEnum::USER_LISTS_TABLE . '.id', '=',
                    ContactsEnum::LIST_CONTACTS_TABLE . '.user_list_id');
                $join->on(ContactsEnum::USER_LISTS_TABLE . '.user_id', '=', ContactsEnum::CONTACTS_TABLE . '.user_id');
            })
            ->whereIn(ContactsEnum::USER_LISTS_TABLE . '.id', $listIds)
            ->groupBy(ContactsEnum::CONTACTS_TABLE . '.phone')
            ->having('contact_optout', true)
            ->get([
                ContactsEnum::CONTACTS_TABLE . '.phone',
                $this->db::raw($this->getCaseWhenString())
            ]);
    }

    /**
     * @param $listIds
     * @param $userId
     * @return array
     */
    public function getOptInContactsByCode($listIds, $userId)
    {
        $sub = $this->db::connection(ContactsEnum::DB_CONNECTION)->table(ContactsEnum::CONTACTS_TABLE)
            ->select([
                ContactsEnum::CONTACTS_TABLE . '.phone',
                ContactsEnum::CONTACTS_TABLE . '.a2_code',
                $this->db::raw($this->getCaseWhenString())
            ])
            ->leftJoin(ContactsEnum::LIST_CONTACTS_TABLE,
                ContactsEnum::CONTACTS_TABLE . '.id', '=', ContactsEnum::LIST_CONTACTS_TABLE . '.contact_id')
            ->leftJoin(ContactsEnum::USER_LISTS_TABLE, function ($join) {
                $join->on(ContactsEnum::USER_LISTS_TABLE . '.id', '=',
                    ContactsEnum::LIST_CONTACTS_TABLE . '.user_list_id');
                $join->on(ContactsEnum::USER_LISTS_TABLE . '.user_id', '=', ContactsEnum::CONTACTS_TABLE . '.user_id');
            })
            ->whereIn(ContactsEnum::USER_LISTS_TABLE . '.id', $listIds)
            ->where(ContactsEnum::USER_LISTS_TABLE . '.user_id', $userId)
            ->groupBy(ContactsEnum::CONTACTS_TABLE . '.phone')
            ->having('contact_optout', 0);
        $contactsQuery = $this->db::connection(ContactsEnum::DB_CONNECTION)->table($this->db::raw("({$sub->toSql()}) as sub"))
            ->mergeBindings($sub)
            ->groupBy('a2_code')
            ->select(['phone', $this->db::raw('COUNT(*) as total_contacts')]);
        return $contactsQuery->pluck('total_contacts', 'phone')->toArray();
    }

    /**
     * @return string
     */
    private function getCaseWhenString(): string
    {
        return 'CASE WHEN ' . ContactsEnum::CONTACTS_TABLE . '.is_optout ' .
            'THEN ' .
            ContactsEnum::CONTACTS_TABLE . '.is_optout ' .
            'ELSE min(' . ContactsEnum::LIST_CONTACTS_TABLE . '.is_optout) ' .
            'END as contact_optout';
    }
}
