<?php

namespace Intergo\OptOut;

use Intergo\OptOut\Enums\OptOutEnum;
use Intergo\OptOut\Repositories\ContactRepository;
use Illuminate\Redis\Connections\Connection;
use Illuminate\Support\Facades\DB;
use Illuminate\Support\Facades\Redis;
use Intergo\Helpers\HashHelper;
use Intergo\OptOut\ValueObjects\OptOutPlaceholderVO;
use Intergo\OptOut\ValueObjects\OptOutVO;
use Intergo\Shortlinks\Enums\ShortlinksEnum;

abstract class AbstractOptOutService
{
    protected Redis $redis;
    protected DB $db;

    /**
     * AbstractOptOutService constructor.
     */
    public function __construct(Redis $redis, DB $db)
    {
        $this->redis = $redis;
        $this->db = $db;
    }

    /**
     * @return string
     */
    protected function cacheConnectionName()
    {
        return OptOutEnum::CACHE_CONNECTION;
    }

    /**
     * @return Connection
     */
    private function cacheConnection(): Connection
    {
        return $this->redis::connection($this->cacheConnectionName());
    }

    /**
     * @param OptOutVO $optOutVO
     * @return void
     */
    public function cacheOptOuts(OptOutVO $optOutVO)
    {
        $this->cacheGlobalOptOuts($optOutVO->authUserId, $optOutVO->appUserId);
        if (empty($optOutVO->listIds)) {
            return;
        }
        $this->cacheListOptOuts($optOutVO->authUserId, $optOutVO->listIds);
    }

    /**
     * @param $userId
     * @param $appUserId
     * @return void
     */
    public function cacheGlobalOptOuts($userId, $appUserId)
    {
        if ($this->existsOnCache($userId, OptOutEnum::GLOBAL_OPT_OUT)) {
            return;
        }
        $optOuts = (new ContactRepository($this->db))->getOptOutsByUserId($appUserId);
        //Cache result to avoid multiple query executions
        $this->setOnCache($userId, OptOutEnum::GLOBAL_OPT_OUT);
        foreach ($optOuts as $optOut) {
            $this->setOnCache($userId, $optOut->phone);
        }
    }

    /**
     * @param $userId
     * @param $listIds
     * @return void
     */
    public function cacheListOptOuts($userId, $listIds)
    {
        if ($this->existsOnCache($userId, OptOutEnum::LIST_OPT_OUT)) {
            return;
        }
        [$listIds, $listKey] = $this->processListIdsAndGetListKey($listIds);
        $optOuts = (new ContactRepository($this->db))->getOptOutsByListIds($listIds);
        //Cache result to avoid multiple query executions
        $this->setOnCache($userId, OptOutEnum::LIST_OPT_OUT);
        foreach ($optOuts as $optOut) {
            $this->setOnCache($userId, self::getListOptOutField($listKey, $optOut->phone));
        }
    }

    /**
     * Check if a number is unsubscribed
     *
     * @param $userId
     * @param $phoneNumber
     * @param $listIds
     * @param ?bool $bypassOptOut
     * @return bool
     */
    public function isUnsubscribed($userId, $phoneNumber, $listIds = null, ?bool $bypassOptOut = false): bool
    {
        if ($bypassOptOut) {
            return false;
        }
        if ($this->isGlobalOptOut($userId, $phoneNumber)) {
            return true;
        }
        if (empty($listIds)) {
            return false;
        }
        [$listIds, $listKey] = $this->processListIdsAndGetListKey($listIds);
        return $this->existsOnCache($userId, self::getListOptOutField($listKey, $phoneNumber));
    }

    /**
     * @param $userId
     * @param $phoneNumber
     * @return boolean
     */
    public function isGlobalOptOut($userId, $phoneNumber): bool
    {
        return $this->existsOnCache($userId, $phoneNumber);
    }

    /**
     * @param $listKey
     * @param $phoneNumber
     * @return string
     */
    public function getListOptOutField($listKey, $phoneNumber): string
    {
        return "$listKey:$phoneNumber";
    }

    /**
     * @param $listIds
     * @return array
     */
    public function processListIdsAndGetListKey($listIds): array
    {
        $listIds = is_array($listIds) ? $listIds : [$listIds];
        sort($listIds);
        $listKey = HashHelper::hash(implode(':', $listIds));
        return [$listIds, $listKey];
    }

    /**
     * @param OptOutVO $optOutVO
     * @return array
     */
    public function getOptInContactsByCode(OptOutVO $optOutVO)
    {
        [$listIds, $listKey] = $this->processListIdsAndGetListKey($optOutVO->listIds);
        $hashKey = OptOutEnum::getListEstimateHashKey($optOutVO->authUserId);
        $result = $this->getFromCache($hashKey, $listKey);
        if ($result) {
            return json_decode($result, true);
        }
        $contacts = (new ContactRepository($this->db))->getOptInContactsByCode($listIds, $optOutVO->appUserId);
        $result = json_encode($contacts);
        $this->setOnCache($hashKey, $listKey, $result);
        return $contacts;
    }

    /**
     * @param $userId
     * @return void
     */
    public function invalidateOptOutFlags($userId)
    {
        foreach (OptOutEnum::getUserHashKeys($userId) as $hashKey) {
            $fields = $this->cacheConnection()->hkeys($hashKey);
            foreach ($fields as $field) {
                $this->cacheConnection()->hdel($hashKey, $field);
            }
        }
    }

    /**
     * Insert optOut links if placeholder exists
     *
     * @param string $message
     * @param string $optOutUrl
     * @return OptOutPlaceholderVO|null
     */
    public static function prepareOptOutLink(string $message, string $optOutUrl)
    {
        if (strpos($message, OptOutEnum::OPT_OUT_PLACEHOLDER) !== false) {
            $optOutPlaceholderVO = new OptOutPlaceholderVO();
            $optOutPlaceholderVO->optOutCode = substr(md5(rand()), 0, 7);
            $optOutUrl = "$optOutUrl/$optOutPlaceholderVO->optOutCode";
            $optOutPlaceholderVO->messageBody = str_replace(OptOutEnum::OPT_OUT_PLACEHOLDER, PHP_EOL . $optOutUrl, $message);
            return $optOutPlaceholderVO;
        }
        return null;
    }

    /**
     * @param string $key
     * @param string $field
     * @param mixed $value
     * @return void
     */
    private function setOnCache(string $key, string $field, $value = true): void
    {
        $this->cacheConnection()->hset(OptOutEnum::cacheKeyUser($key), $field, $value);
    }

    /**
     * @param $key
     * @param $field
     * @return bool
     */
    private function existsOnCache($key, $field): bool
    {
        return $this->cacheConnection()->hexists(OptOutEnum::cacheKeyUser($key), $field);
    }

    /**
     * @param $key
     * @param $field
     * @return mixed
     */
    private function getFromCache($key, $field)
    {
        return $this->cacheConnection()->hget(OptOutEnum::cacheKeyUser($key), $field);
    }
}
