<?php

namespace Intergo\MessageClient;

use Intergo\BaseClient\Client;
use Intergo\BaseClient\Responses\BaseResponseInterface;
use Intergo\MessageClient\ValueObject\ProviderErrorCodeSearchVO;

class AbstractMessageService
{
    protected MessageClient $client;

    /**
     * AbstractMessageService constructor.
     *
     * @param $baseUrl
     * @param $traceId
     */
    public function __construct($baseUrl, $traceId)
    {
        $this->client = new MessageClient($baseUrl, $traceId);
    }

    /**
     * @param string $user_id
     * @param $settings
     * @return BaseResponseInterface
     */
    public function updateUserSettingsV2(string $user_id, $settings)
    {
        return $this->client->execute(__FUNCTION__, compact('user_id'), compact('settings'));
    }

    /**
     * @param string $user_id
     * @param $viber_settings
     * @param $viber_prices
     * @return BaseResponseInterface
     */
    public function updateViberSettings(string $user_id, $viber_settings, $viber_prices)
    {
        return $this->client->execute(__FUNCTION__, compact('user_id'), compact('viber_settings', 'viber_prices'));
    }

    /**
     * @param string $user_id
     * @return BaseResponseInterface
     */
    public function getUserSettings(string $user_id)
    {
        return $this->client->execute(__FUNCTION__, compact('user_id'), []);
    }

    /**
     * @param $user_id
     * @return BaseResponseInterface
     */
    public function invalidateUserLists($user_id)
    {
        return $this->client->execute(__FUNCTION__, compact('user_id'), []);
    }

    /**
     * @param $id
     * @param $data
     * @return BaseResponseInterface
     */
    public function updateNetwork($id, $data)
    {
        extract($data);
        unset($data);
        return $this->client->execute(__FUNCTION__, compact('id'), get_defined_vars());
    }

    /**
     * @param $message_id
     * @return BaseResponseInterface
     */
    public function updateStatusMessageFromProvider($message_id): BaseResponseInterface
    {
        return $this->client->execute(__FUNCTION__, compact('message_id'), get_defined_vars());
    }

    /**
     * @param $from
     * @param $to
     * @param $provider_id
     * @param $user_id
     * @return BaseResponseInterface
     */
    public function updateStatusMessageMassFromProvider($from, $to, $provider_id, $user_id, $bearerToken): BaseResponseInterface
    {
        return $this->client->executeAuth(__FUNCTION__, $bearerToken, [],
            compact("from", "to", "provider_id", "user_id"));
    }

    /**
     * @return BaseResponseInterface
     */
    public function getCountryRatesAll()
    {
        return $this->client->execute(__FUNCTION__);
    }

    /**
     * @param $a2_code
     * @return BaseResponseInterface
     */
    public function getCountryRate($a2_code)
    {
        return $this->client->execute(__FUNCTION__, ['id' => $a2_code]);
    }

    /**
     * @param $a2_code
     * @param $client_rate
     * @return BaseResponseInterface
     */
    public function updateCountryRate($a2_code, $client_rate, $bearerToken)
    {
        return $this->client->executeAuth(__FUNCTION__, $bearerToken, [], compact('a2_code', 'client_rate'));
    }

    /**
     * @param $a2_code
     * @return BaseResponseInterface
     */
    public function deleteCountryRate($a2_code, $bearerToken)
    {
        return $this->client->executeAuth(__FUNCTION__, $bearerToken, ['id' => $a2_code]);
    }

    /**
     * @param ProviderErrorCodeSearchVO $search
     * @param int $page
     * @return BaseResponseInterface
     */
    public function getProviderErrorCodes(ProviderErrorCodeSearchVO $search, int $page = 1)
    {
        $state = $search->state;
        $reason_code = $search->reason_code;

        return $this->client->execute(__FUNCTION__, [], compact('state', 'reason_code', 'page'));
    }

    /**
     * @param $id
     * @return BaseResponseInterface
     */
    public function getProviderErrorCode($id)
    {
        return $this->client->execute(__FUNCTION__, compact('id'));
    }

    /**
     * @param array $data
     * @return BaseResponseInterface
     */
    public function createProviderErrorCode(array $data)
    {
        return $this->client->execute(__FUNCTION__, null, $data);
    }

    /**
     * @param $id
     * @param array $data
     * @return BaseResponseInterface
     */
    public function updateProviderErrorCode($id, array $data)
    {
        return $this->client->execute(__FUNCTION__, compact('id'), $data);
    }

    /**
     * @param $id
     * @return BaseResponseInterface
     */
    public function deleteProviderErrorCode($id)
    {
        return $this->client->execute(__FUNCTION__, compact('id'));
    }
}
