<?php

namespace Intergo\Log;

use Intergo\Log\Contracts;
use Intergo\Log\Exceptions;
use Psr\Log\LoggerInterface;

/**
 * @method info($message, $data = null)
 * @method error($message, $data = null)
 * @method emergency($message, $data = null)
 * @method alert($message, $data = null)
 * @method critical($message, $data = null)
 * @method warning($message, $data = null)
 * @method notice($message, $data = null)
 * @method debug($message, $data = null)
 */
class GrayLogHandler
{

    /**
     * @var string
     */
    private $trace_id = '';

    /**
     * @var string[]
     */
    private $sensitive_data = ['password'];

    /**
     * @var string
     */
    private $tag = '';

    /**
     * @var LoggerInterface
     */
    private $log;

    /**
     * @var Contracts\EnumContract
     */
    private $fieldEnums;

    /**
     * @var string[]
     */
    private static $levels = [
        'emergency',
        'alert',
        'critical',
        'error',
        'warning',
        'notice',
        'info',
        'debug'
    ];

    /**
     * GrayLoggerHandler constructor.
     */
    public function __construct(LoggerInterface $log, string $fieldEnums)
    {
        $this->log        = $log;
        $this->fieldEnums = $fieldEnums;
    }

    /**
     * @param $trace_id
     * @return GrayLogHandler
     */
    public function setTraceId($trace_id): GrayLogHandler
    {
        $this->trace_id = $trace_id;
        return $this;
    }

    /**
     * @return string
     */
    public function getTraceId(): string
    {
        return $this->trace_id;
    }

    /**
     * @param Contracts\EnumContract $fieldEnums
     * @return GrayLogHandler
     */
    public function setFieldEnum(Contracts\EnumContract $fieldEnums): GrayLogHandler
    {
        $this->fieldEnums = $fieldEnums;
        return $this;
    }

    /**
     * @param mixed $tag
     */
    public function setTag($tag): GrayLogHandler
    {
        $this->tag = $tag;
        return $this;
    }

    /**
     * @param $name
     * @param $arguments
     * @throws Exceptions\InvalidLogFieldContract
     */
    public function __call($name, $arguments)
    {
        if (empty($this->fieldEnums)) {
            throw new Exceptions\InvalidLogFieldContract('No enum fields provided');
        }
        if (!in_array($name, self::$levels)) {
            return;
        }

        $data = [];
        //we assume the first string we find is the message to be send
        $message = array_shift($arguments) ?? "";

        //use this count to add custom_fields eg. custom_field_1
        $count = 1;

        foreach ($arguments as $argument) {
            if (is_array($argument)) {
                $data = array_merge($data, $argument);
                continue;
            }
            if (!is_string($argument)) {
                $argument = json_encode($argument);
            }
            $data = array_merge(["custom_field_$count" => $argument], $data);
            $count++;
        }
        $final_data             = $this->fieldEnums::typeCastValues($data);
        $final_data['trace_id'] = $this->trace_id;
        $final_data['tag']      = $this->tag;
        $this->clearDataFromSensitiveInformation($final_data);
        $this->log->$name($message, $final_data);
    }

    /**
     * Unset any sensitive data
     * @param $data
     */
    private function clearDataFromSensitiveInformation(&$data)
    {
        $keys = array_keys($data);
        foreach ($this->sensitive_data as $sensitive_datum) {
            if (in_array($sensitive_datum, $keys)) {
                unset($data[$sensitive_datum]);
            }
        }
    }
}
