<?php


namespace Intergo\Log\Contracts;


use ReflectionClass;

abstract class AbstractEnumContract implements EnumContract
{
    /**
     * Overload the below functions to get your fields
     * hint: In case of multi-inheritance use
     * return array_merge(parent::getIntegerFields, static::your_local_array);
     */
    protected static function getIntegerFields(): array
    {
        return static::$integer_fields;
    }

    protected static function getFloatFields(): array
    {
        return static::$float_fields;
    }

    protected static function getStringFields(): array
    {
        return static::$string_fields;
    }

    protected static function getBooleanFields(): array
    {
        return static::$boolean_fields;
    }

    protected static function getMappingFields(): array
    {
        return static::$mapping_fields;
    }

    /**
     * Below private arrays are being used only for caches purposes in order to avoid the cost of calling one of
     * getFields functions that may include merge
     */
    private static array $mapping_fields = [];

    private static array $integer_fields = [];

    private static array $float_fields = [];

    private static array $string_fields = [];

    private static array $boolean_fields = [];

    /**
     * This method will check if the field exist as a constant above.
     * @param $field
     * @return bool
     */
    public static function doesFieldExist($field): bool
    {
        $oClass    = new ReflectionClass(__CLASS__);
        $constants = array_values($oClass->getConstants());
        if (in_array($field, $constants)) {
            return true;
        }
        return false;
    }

    /**
     * Use this method to get the correct typeCast value so we keep the consistency in Graylog
     * @param $data
     * @return array
     */
    public static function typeCastValues($data): array
    {
        if (empty(static::$string_fields)) {
            /**
             * I make the assumption that if 1 of the arrays is empty then the caching arrays never initialized
             * i am checking only the string_fields array to avoid having 4 different same if statements
             * this has an issue in case that the enum is not having any string value.
             */
            static::$string_fields  = static::getStringFields();
            static::$float_fields   = static::getFloatFields();
            static::$integer_fields = static::getIntegerFields();
            static::$mapping_fields = static::getMappingFields();
            static::$boolean_fields = static::getBooleanFields();
        }

        $final_data = [];
        foreach ($data as $field => $value) {
            $field = static::getMappingField($field);
            if (is_numeric($field)) {
                $field = 'array_index_' . $field;
            }
            if (in_array($field, static::$integer_fields)) {
                $final_data[$field] = intval($value);
            } elseif (in_array($field, static::$float_fields)) {
                $final_data[$field] = floatval($value);
            } elseif (in_array($field, static::$string_fields)) {
                $final_data[$field] = $value;
            } elseif (in_array($field, static::$boolean_fields)) {
                $result = ((bool)$value) ? "true" : "false";
                if ($value === "false") {
                    $result = "false";
                }
                $final_data[$field] = $result;
            } elseif (is_array($value)) {
                $final_data[$field] = json_encode($value);
            } else {
                $is_value_json_decoded = @json_decode($value);
                if (!is_object($is_value_json_decoded)) {
                    if (!isset($final_data['extra_params'])) {
                        $final_data['extra_params'] = "";
                    }
                    $final_data['extra_params'] .= json_encode([$field => $value]);
                }
            }
        }
        return $final_data;
    }

    /**
     * Use this method to get the relevant mapping field based on the different variations we have on code
     * @param $field
     * @return string
     */
    public static function getMappingField($field): string
    {
        return static::$mapping_fields[$field] ?? $field;
    }
}
