## Usage

Services implementing PSR Log interface `LoggerInterface`, can use this package. Different setup of this package is
based on the frameworks used.

### Spiral Framework

By default, initial setup of the framework doesn't implement log. To setup the package on spiral, we need to follow as
provided

#### Add Bootloaders

```php

// LoggingBootloader.php

<?php

/**
 * This file is part of Spiral package.
 *
 * For the full copyright and license information, please view the LICENSE
 * file that was distributed with this source code.
 */

declare(strict_types=1);

namespace App\Bootloader;

use Gelf\Publisher;
use Gelf\Transport\UdpTransport;
use Monolog\Formatter\GelfMessageFormatter;
use Monolog\Formatter\LineFormatter;
use Monolog\Formatter\NormalizerFormatter;
use Monolog\Handler\GelfHandler;
use Monolog\Handler\HandlerInterface;
use Monolog\Handler\RotatingFileHandler;
use Monolog\Handler\StreamHandler;
use Monolog\Logger;
use Psr\Log\LoggerInterface;
use Spiral\Boot\Bootloader\Bootloader;
use Spiral\Config\ConfiguratorInterface;
use Spiral\Config\Patch\Append;
use Spiral\Core\Container;
use Spiral\Http\Middleware\ErrorHandlerMiddleware;
use Spiral\Logger\LogsInterface;
use Spiral\Monolog\LogFactory;

class LoggingBootloader extends Bootloader implements Container\SingletonInterface
{
    protected const SINGLETONS = [
        LogsInterface::class => LogFactory::class,
        LoggerInterface::class => Logger::class,
    ];

    /** @var ConfiguratorInterface */
    private $config;

    private $logConfig;

    /**
     * Bootloader execute method.
     *
     * @param Container $container
     * @param ConfiguratorInterface $config
     */
    public function boot(Container $container, ConfiguratorInterface $config): void
    {
        $this->config = $config;
        $container->bindInjector(Logger::class, LogFactory::class);
        $logConfig = $this->config->getConfig('log');
        $defaultLog = $logConfig['default'];
        $this->logConfig = $logConfig['drivers'][$defaultLog];
        switch ($defaultLog) {
            case "daily":
                $this->logToFile();
                break;
            case "stack":
                $this->logToStack();
                break;
            default:
                $this->logToConsole();
                break;
        }
    }

    /**
     * @param string $channel
     * @param HandlerInterface $handler
     */
    public function addHandler(string $channel, HandlerInterface $handler): void
    {
        if (!isset($this->config->getConfig('monolog')['handlers'][$channel])) {
            $this->config->modify('monolog', new Append('handlers', $channel, []));
        }

        $this->config->modify('monolog', new Append(
            'handlers.' . $channel,
            null,
            $handler
        ));
    }

    /**
     * @param string $filename
     * @param NormalizerFormatter $formatter
     * @param int $level
     * @param int $maxFiles
     * @param bool $bubble
     * @return HandlerInterface
     */
    public function logRotate(string $filename, NormalizerFormatter $formatter, int $level = Logger::DEBUG, int $maxFiles = 0, bool $bubble = false): HandlerInterface
    {
        $handler = new RotatingFileHandler(
            $filename,
            $maxFiles,
            $level,
            $bubble,
            null,
            true
        );
        return $this->formatLog($formatter, $handler);
    }

    public function console(NormalizerFormatter $formatter, int $level = Logger::DEBUG, bool $bubble = false)
    {
        $handler = new StreamHandler(STDERR, $level, $bubble);
        return $this->formatLog($formatter, $handler);
    }

    private function formatLog($formatter, $handler)
    {
        if ($formatter instanceof LineFormatter) {
            return $handler->setFormatter(
                new $formatter("[%datetime%] %level_name%: %message% %context%\n")
            );
        }
        return $handler->setFormatter(
            new $formatter()
        );
    }

    private function logToFile()
    {
        // http level errors
        $this->addHandler(ErrorHandlerMiddleware::class, $this->logRotate(
            $this->logConfig["directory"] . '/http.log',
            new $this->logConfig['formatter']()
        ));

        // app level errors
        $this->addHandler(LogFactory::DEFAULT, $this->logRotate(
            $this->logConfig["directory"] . '/error.log',
            new $this->logConfig['formatter'](),
            Logger::ERROR,
            25,
            false
        ));

        // debug and info messages via global LoggerInterface
        $this->addHandler(LogFactory::DEFAULT, $this->logRotate(
            $this->logConfig["directory"] . '/debug.log',
            new $this->logConfig['formatter']()
        ));
    }

    private function logToConsole()
    {
        $this->addHandler(ErrorHandlerMiddleware::class, $this->console(
            new $this->logConfig['formatter']()
        ));
        $this->addHandler(LogFactory::DEFAULT, $this->console(
            new $this->logConfig['formatter']()
        ));
    }

    private function logToStack()
    {
        $server = $this->logConfig["host"];
        $port = $this->logConfig["port"];
        $transport = new UdpTransport($server, $port);
        $publisher = new Publisher($transport);
        $gelfHandler = new GelfHandler($publisher);
        $formatter = new GelfMessageFormatter(null, null, "");
        $gelfHandler->setFormatter($formatter);
        $this->addHandler(LogFactory::DEFAULT, $gelfHandler);
    }
}

```

#### Extend AbstractEnumContract and name it LoggingFieldsEnum

```php
<?php

namespace App\Enums;

use Intergo\Log\Contracts\AbstractEnumContract;

class LoggingFieldsEnum extends AbstractEnumContract
{
    const IS_RENTED = "is_rented";
    const IS_SHARED = "is_shared";
    const NUMBER = "auth_user_id";
    const VENDOR_ID = "callback_url";

    //add here only values that are integers
    public static array $integer_fields = [
        self::IS_RENTED,
        self::IS_SHARED,
        self::VENDOR_ID
    ];

    //add here only values that are pure strings
    public static array $string_fields = [
        self::NUMBER
    ];

}
```

```php

// GraylogBootloader.php
<?php


namespace App\Bootloader;


use App\Enums\LoggingFieldsEnum;
use App\Middleware\GraylogMiddleware;
use Intergo\Log\GrayLogHandler;
use Psr\Log\LoggerInterface;
use Spiral\Boot\Bootloader\Bootloader;
use Spiral\Bootloader\Http\HttpBootloader;
use Spiral\Core\Container\SingletonInterface;

class GraylogBootloader extends Bootloader implements SingletonInterface
{
    protected const SINGLETONS = [
        GrayLogHandler::class => [self::class, 'log']
    ];

    private function log(): GrayLogHandler
    {
        return new GrayLogHandler(spiral(LoggerInterface::class), new LoggingFieldsEnum());
    }

    public function boot(HttpBootloader $http)
    {
        $http->addMiddleware(GraylogMiddleware::class);
    }
}

```

Add Bootloaders in `App.php`

```php
<?php

/**
 * This file is part of Spiral package.
 *
 * For the full copyright and license information, please view the LICENSE
 * file that was distributed with this source code.
 */

declare(strict_types=1);

namespace App;

class App extends Kernel
{
    /*
     * List of components and extensions to be automatically registered
     * within system container on application start.
     */
    protected const LOAD = [
        // Base extensions
        Monolog\MonologBootloader::class,
        Bootloader\LoggingBootloader::class,
    ];

    /*
     * Application specific services and extensions.
     */
    protected const APP = [
        Bootloader\GraylogBootloader::class,
    ];
}

```

##### Logging Example

```php
<?php
$payload = [
    LoggingFieldsEnum::NUMBER => '+3579912335564',
    LoggingFieldsEnum::VENDOR_ID => 1,
    LoggingFieldsEnum::IS_RENTED => 1,
    LoggingFieldsEnum::IS_SHARED => 0,
];
spiral(GrayLogHandler::class)
    ->setTag(LoggingTagsEnum::ADMIN_INVENTORY_ADD)
    ->info('Admin Activities - Add Inventory', $payload);
```

### Laravel Framework

#### There are 2 options to use this package.

1. Option 1: Use it as it is and not add new fields. The process is:
   - Set your GelfProcessor in case you want to inject user details in the logs eg
```php
<?php

namespace App\Services;

use App\Enums\LoggingFieldsEnum;
use Illuminate\Auth\AuthServiceProvider;

class GelfProcessor extends \Intergo\Log\LaravelHelpers\Services\GelfProcessor
{

    public function getUser()
    {
        if (\App::providerIsLoaded(AuthServiceProvider::class)) {
            return auth()->user();
        }
        return null;
    }
}
   
```
  - Define your custom GelfProcessor under `config/logging.php` under gelf processors section eg.

```php

        'gelf' => [
            'driver' => 'custom',

            'via' => \Hedii\LaravelGelfLogger\GelfLoggerFactory::class,

            // This optional option determines the processors that should be
            // pushed to the handler. This option is useful to modify a field
            // in the log context (see NullStringProcessor), or to add extra
            // data. Each processor must be a callable or an object with an
            // __invoke method: see monolog documentation about processors.
            // Default is an empty array.
            'processors' => [
                Hedii\LaravelGelfLogger\Processors\NullStringProcessor::class,
                App\Services\GelfProcessor::class,
            ],
```
  - You are ready to use now the graylog

2. Option 2: you have some custom fields to add on your own LoggingFields Enum. The process is:
   - Create a new LoggingFieldsEnum and extend the LoggingFieldsEnum from package eg.

```php
<?php

namespace App\Enums;

use App\Enums\MyApplicationFieldsEnum;

class AnExtraFieldsEnum extends \Intergo\Log\Contracts\LoggingFieldsEnum
{
    const IS_RENTED2 = "is_rented2";
    const IS_SHARED2 = "is_shared2";
    const NUMBER2 = "auth_user_id2";
    const VENDOR_ID2 = "callback_url2";

    //add here only values that are integers
    protected static function getIntegerFields(): array
    {
        return array_merge(parent::getIntegerFields(), [
            self::IS_RENTED2,
            self::IS_SHARED2,
            self::VENDOR_ID2
        ]);
    }       

    //add here only values that are pure strings
    protected static function getStringFields(): array
    {
        return array_merge(parent::getStringFields(), [
            self::NUMBER2,
        ]);
    }
}

```
   - Add the GraylogServiceProvider and extend the GraylogServiceProvider from package to define the custom LoggingFieldsEnum class eg.
```php
<?php
// GraylogServiceProvider.php

namespace App\Providers;


<?php

namespace App\Providers;

use App\Enums\LoggingFieldsEnum;

class GraylogServiceProvider extends \Intergo\Log\LaravelHelpers\Providers\GraylogServiceProvider
{
    protected static string $LoggingFieldsClassName = LoggingFieldsEnum::class;
}

```

  - Add the Custom Provider under config/app.php.
  - Define also GelfProcessor to set some user fields.
  - You are ready to use now the graylog


### Logging Example

```php
<?php
$payload = [
    LoggingFieldsEnum::NUMBER => '+3579912335564',
    LoggingFieldsEnum::VENDOR_ID => 1,
    LoggingFieldsEnum::IS_RENTED => 1,
    LoggingFieldsEnum::IS_SHARED => 0,
];
app(GrayLogHandler::class)
    ->setTag(LoggingTagsEnum::ADMIN_INVENTORY_ADD)
    ->info('Admin Activities - Add Inventory', $payload);
```