<?php
declare(strict_types=1);

namespace Intergo\LaravelAuthMiddleware\Services\Microservices;

use Intergo\AuthClient\AbstractAuthService;
use Intergo\BaseClient\Responses\BaseResponseInterface;
use Intergo\LaravelAuthMiddleware\Events\UserLoggedIn;
use Intergo\Log\Contracts\LoggingFieldsEnum;
use Intergo\Log\GrayLogHandler;

class AuthService extends AbstractAuthService
{
    const SUCCESS = 'success';
    const FAILED = 'failed';
    const SESSION_JWT_TOKEN = 'auth_token';
    const SESSION_AUTH_USER = 'auth_user';
    const DEFAULT_IP = '0.0.0.0';
    private GrayLogHandler $grayLogger;

    /**
     * AuthService constructor.
     */
    public function __construct()
    {
        $this->grayLogger = app(GrayLogHandler::class);

        parent::__construct(
            config('auth-middleware.site'),
            config('auth-middleware.url'),
            config('auth-middleware.jwt_secret'),
            $this->grayLogger->getTraceId()
        );
    }

    /**
     * @param $site
     * @param $secret
     * @return AuthService
     */
    public function setCredentials($site, $secret)
    {
        $this->site = $site;
        $this->secret = $secret;
        return $this;
    }

    /**
     * @param string $action
     * @param $ip
     * @param BaseResponseInterface $result
     * @param ?string $email
     * @return BaseResponseInterface
     */
    public function response(string $action, $ip, BaseResponseInterface $result, ?string $email = null)
    {
        if ($result->isSuccessful()) {
            $auth_result = self::SUCCESS;
            $this->grayLogger->info("{$action}:{$auth_result}", [
                LoggingFieldsEnum::AUTH_RESULT => $auth_result,
                LoggingFieldsEnum::IP_ADDRESS => $ip,
            ]);
        } else {
            $auth_result = self::FAILED;
            $this->grayLogger->error("{$action}:{$auth_result}", [
                LoggingFieldsEnum::AUTH_RESULT => $auth_result,
                LoggingFieldsEnum::IP_ADDRESS => $ip,
                LoggingFieldsEnum::EMAIL => $email,
                LoggingFieldsEnum::COUNTRY => $this->getCountryNameByIp($ip),
                LoggingFieldsEnum::RESPONSE_MESSAGE => $result->getMessage(),
                LoggingFieldsEnum::RESPONSE_DATA => $result->getData()['info'] ?? $result->getData(),
                LoggingFieldsEnum::RESPONSE_CODE => $result->getData()['errorCode'] ?? ''
            ]);
        }
        return $result;
    }

    /**
     * @param $authToken
     * @param $authUser
     * @return void
     */
    public function storeJWT($authToken, $authUser): void
    {
        UserLoggedIn::dispatch($authToken, $authUser);
    }

    /**
     * @param $ip
     * @return string
     */
    protected function getCountryNameByIp($ip): string
    {
        return 'N/A';
    }
}
