<?php
declare(strict_types=1);

namespace Intergo\LaravelAuthMiddleware\Http\Middleware\Api;

use Closure;
use App\Models\User;
use Illuminate\Http\JsonResponse;
use Illuminate\Http\Request;
use Intergo\BaseClient\Responses\BaseResponseInterface;
use Intergo\LaravelAuthMiddleware\Http\Middleware\AbstractAuthMiddleware;
use Intergo\LaravelAuthMiddleware\Services\Microservices\AuthService;
use Intergo\LaravelAuthMiddleware\Traits\ResponseTrait;

/**
 * Purpose: Authenticate Api tokens on services.
 *
 * Used by services that don't support Auth Facade, so we merge the user in the request.
 */
class AuthenticateApi extends AbstractAuthMiddleware
{
    use ResponseTrait;

    /**
     * Handle an incoming request.
     *
     * @param Request $request
     * @param Closure $next
     * @param string $params
     * @return JsonResponse|mixed
     */
    public function handle(Request $request, Closure $next, string $params = '')
    {
        $result = parent::handle($request, $next);
        if (!$result->isSuccessful()) {
            return $this->failed('Invalid API Key or Token', [], $result->getStatusCode());
        }
        return $next($this->request);
    }

    /**
     * @return string|null
     */
    protected function getToken()
    {
        return $this->request->bearerToken() ?? $this->request->api_key;
    }

    /**
     * @param string $token
     * @param string $ip
     * @return BaseResponseInterface
     */
    protected function authenticate(string $token, string $ip)
    {
        return app(AuthService::class)->authenticateApi($token, $ip);
    }

    /**
     * @param $user
     * @return void
     */
    protected function loginUser($user)
    {
        $user['_id'] = $user['id'];
        $userCollection = User::hydrate(["user" => $user]);
        $user = $userCollection->first();
        $this->request->user = $user;
    }
}

