<?php
declare(strict_types=1);

namespace Intergo\LaravelAuthMiddleware\Http\Middleware;

use Closure;
use Illuminate\Http\JsonResponse;
use Illuminate\Http\Request;
use Intergo\BaseClient\Responses\BaseResponse;
use Intergo\BaseClient\Responses\BaseResponseInterface;
use Intergo\LaravelAuthMiddleware\Services\Microservices\AuthService;
use Intergo\LaravelAuthMiddleware\Traits\ResponseTrait;
use Intergo\Log\GrayLogHandler;

abstract class AbstractAuthMiddleware
{
    use ResponseTrait;

    protected Request $request;

    /**
     * @return ?string
     */
    abstract protected function getToken();

    /**
     * @param string $token
     * @param string $ip
     * @return BaseResponseInterface
     */
    abstract protected function authenticate(string $token, string $ip);

    /**
     * @param $user
     * @return void
     */
    abstract protected function loginUser($user);

    /**
     * @param Request $request
     * @param Closure $next
     * @param string $params
     * @return BaseResponseInterface
     */
    public function handle(Request $request, Closure $next, string $params = '')
    {
        $this->setRequest($request);
        $token = $this->getToken();
        if (empty($token)) {
            $message = 'AuthMiddleware:Token is missing';
            app(GrayLogHandler::class)->setTag(config('auth-middleware.site'))->error($message);
            return new BaseResponse(['success' => false, 'message' => $message, 'statusCode' => 401]);
        }
        //the default IP is in cases we can't determine the api from the request
        $ip = empty($this->request->ip()) ? AuthService::DEFAULT_IP : $this->request->ip();
        $result = $this->authenticate($token, $ip);
        if ($result->isSuccessful()) {
            $this->loginUser($result->getData());
        }
        return $result;
    }

    /**
     * @param Request $request
     */
    protected function setRequest(Request $request)
    {
        $this->request = $request;
    }
}
