<?php

namespace Intergo\InventoryClient;

use Intergo\BaseClient\Responses\BaseResponseInterface;

abstract class AbstractInventoryService
{
    private InventoryClient $client;
    protected string $secret;
    protected string $site;

    /**
     * InventoryService constructor.
     *
     * @param $baseUrl
     * @param $traceId
     * @param $site
     * @param $secret
     */
    public function __construct($baseUrl, $traceId, $site, $secret)
    {
        $this->secret = $secret;
        $this->site = $site;
        $this->client = new InventoryClient($baseUrl, $traceId);
    }

    /**
     * @param $endpoint
     * @param array $urlParams
     * @param array $bodyParams
     * @return BaseResponseInterface
     */
    protected function executeAuth($endpoint, array $urlParams = [], array $bodyParams = [])
    {
        $urlParams['site'] = $this->site;
        $result = $this->client->executeAuth($endpoint, $this->secret, $urlParams, $bodyParams);
        return $this->response($endpoint, $result);
    }

    /**
     * @param $endpoint
     * @param array $urlParams
     * @param array $bodyParams
     * @return BaseResponseInterface
     */
    protected function execute($endpoint, array $urlParams = [], array $bodyParams = [])
    {
        $urlParams['site'] = $this->site;
        $result = $this->client->execute($endpoint, $urlParams, $bodyParams);
        return $this->response($endpoint, $result);
    }

    /**
     * @param string $action
     * @param BaseResponseInterface $result
     * @return BaseResponseInterface
     */
    abstract public function response(string $action, BaseResponseInterface $result);

    /**
     * @return BaseResponseInterface
     */
    public function getAllCountries(): BaseResponseInterface
    {
        return $this->executeAuth(__FUNCTION__);
    }

    /**
     * @param $code
     * @return BaseResponseInterface
     */
    public function getCountryCities($code): BaseResponseInterface
    {
        return $this->executeAuth(__FUNCTION__, compact('code'));
    }

    /**
     * @return BaseResponseInterface
     */
    public function getUSStates(): BaseResponseInterface
    {
        return $this->executeAuth(__FUNCTION__);
    }

    /**
     * @param $code
     * @return BaseResponseInterface
     */
    public function getStateCities($code): BaseResponseInterface
    {
        return $this->executeAuth(__FUNCTION__, compact('code'));
    }

    /**
     * @return BaseResponseInterface
     */
    public function getInventoryAvailableCountries(): BaseResponseInterface
    {
        return $this->executeAuth(__FUNCTION__);
    }

    /**
     * @param $country_code
     * @param $user_id
     * @param $type_id
     * @return BaseResponseInterface
     */
    public function getNumbersForUser($country_code, $user_id, $type_id): BaseResponseInterface
    {
        return $this->executeAuth(__FUNCTION__, [], compact('country_code', 'user_id', 'type_id'));
    }

    /**
     * @return BaseResponseInterface
     */
    public function getNumberTypes(): BaseResponseInterface
    {
        return $this->executeAuth(__FUNCTION__);
    }

    /**
     * @param $number_id
     * @param $user_id
     * @param $subscription_plan_id
     * @param ?string $keyword
     * @return BaseResponseInterface
     */
    public function requestNumber($number_id, $user_id, $subscription_plan_id, string $keyword = null): BaseResponseInterface
    {
        return $this->executeAuth(__FUNCTION__, [], compact('number_id', 'user_id', 'subscription_plan_id', 'keyword'));
    }

    /**
     * @param $number_id
     * @param $settings
     * @return BaseResponseInterface
     */
    public function updateNumberSettings($number_id, $settings): BaseResponseInterface
    {
        return $this->executeAuth(__FUNCTION__, [], compact('number_id', 'settings'));
    }

    /**
     * @param $user_id
     * @return BaseResponseInterface
     */
    public function getUserRentedNumbers($user_id)
    {
        return $this->executeAuth(__FUNCTION__, [], compact('user_id'));
    }


    /**
     * @param $number_id
     * @param $subscription_id
     * @return BaseResponseInterface
     */
    public function getRentedNumberDetails($number_id, $subscription_id)
    {
        return $this->executeAuth(__FUNCTION__, [], compact('number_id', 'subscription_id'));
    }

    /**
     * @param $number
     * @return BaseResponseInterface
     */
    public function getRentedNumberDetailsByNumber($number)
    {
        return $this->executeAuth(__FUNCTION__, [], compact('number'));
    }

    /**
     * @param $subscription_id
     * @return BaseResponseInterface
     */
    public function cancelNumberSubscription($subscription_id)
    {
        return $this->executeAuth(__FUNCTION__, [], compact('subscription_id'));
    }

    /**
     * @param $number_id
     * @param $keyword
     * @return BaseResponseInterface
     */
    public function validateKeyword($number_id, $keyword)
    {
        return $this->executeAuth(__FUNCTION__, [], compact('number_id', 'keyword'));
    }

    /**
     * @param $request_id
     * @return BaseResponseInterface
     */
    public function cancelKeywordRequest($request_id)
    {
        return $this->executeAuth(__FUNCTION__, compact('request_id'));
    }

    /**
     * @param $group_plan_id
     * @param $settings
     * @return BaseResponseInterface
     */
    public function updateGroupPlanSettings($group_plan_id, $settings): BaseResponseInterface
    {
        return $this->executeAuth(__FUNCTION__, [], compact('group_plan_id', 'settings'));
    }
}
