<?php

namespace Intergo\InventoryClient;

use Intergo\BaseClient\Responses\BaseResponseInterface;

abstract class AbstractInventoryService
{
    private InventoryClient $client;

    /**
     * InventoryService constructor.
     *
     * @param $baseUrl
     * @param $traceId
     */
    public function __construct($baseUrl, $traceId)
    {
        $this->client = new InventoryClient($baseUrl, $traceId);
    }

    /**
     * @param $endpoint
     * @param array $urlParams
     * @param array $bodyParams
     * @return BaseResponseInterface
     */
    protected function execute($endpoint, array $urlParams = [], array $bodyParams = [])
    {
        $result = $this->client->execute($endpoint, $urlParams, $bodyParams);
        return $this->response($endpoint, $result);
    }

    /**
     * @param string $action
     * @param BaseResponseInterface $result
     * @return BaseResponseInterface
     */
    abstract public function response(string $action, BaseResponseInterface $result);

    /**
     * @param $number
     * @return BaseResponseInterface
     */
    public function getNumberSubscriptionDetails($number)
    {
        return $this->execute(__FUNCTION__, compact('number'));
    }

    /**
     * @param $id
     * @return BaseResponseInterface
     */
    public function getUserSubscriptions($id)
    {
        return $this->execute(__FUNCTION__, compact('id'));
    }

    /**
     * @param $id
     * @param $subscription_id
     * @return BaseResponseInterface
     */
    public function getUserSubscriptionDetails($id, $subscription_id)
    {
        return $this->execute(__FUNCTION__, compact('id', 'subscription_id'));
    }

    /**
     * @param $id
     * @param $subscription_id
     * @param $log_id
     * @return BaseResponseInterface
     */
    public function getUserSubscriptionBillingLogDetails($id, $subscription_id, $log_id)
    {
        return $this->execute(__FUNCTION__, compact('id', 'subscription_id', 'log_id'));
    }

    /**
     * @param $id
     * @param $subscription_id
     * @return BaseResponseInterface
     */
    public function cancelUserSubscription($id, $subscription_id)
    {
        return $this->execute(__FUNCTION__, compact('id', 'subscription_id'));
    }

    /**
     * @param $id
     * @param $subscription_id
     * @return BaseResponseInterface
     */
    public function renewUserSubscription($id, $subscription_id)
    {
        return $this->execute(__FUNCTION__, compact('id', 'subscription_id'));
    }

    /**
     * @param $id
     * @param $type
     * @param $grace_period
     * @return BaseResponseInterface
     */
    public function subscriptionOnGraceCallback($id, $type, $grace_period)
    {
        return $this->execute(
            __FUNCTION__,
            compact('id', 'type'),
            compact('grace_period')
        );
    }

    /**
     * @param $id
     * @param $type
     * @param $setup_fee
     * @param $recurring_fee
     * @param $commit_period
     * @return BaseResponseInterface
     */
    public function subscriptionRenewalCallback($id, $type, $setup_fee, $recurring_fee, $commit_period)
    {
        return $this->execute(
            __FUNCTION__,
            compact('id', 'type'),
            compact('setup_fee', 'recurring_fee', 'commit_period')
        );
    }

    /**
     * @param $id
     * @param $type
     * @return BaseResponseInterface
     */
    public function subscriptionTerminationCallback($id, $type)
    {
        return $this->execute(__FUNCTION__, compact('id', 'type'));
    }

    /**
     * @param $type
     * @param $subscription_ids
     * @param $days
     * @return BaseResponseInterface
     */
    public function subscriptionNumberExpiringCallback($type, $subscription_ids, $days)
    {
        return $this->execute(__FUNCTION__, compact('type'), compact('subscription_ids', 'days'));
    }

    /**
     * @param $user_id
     * @param $email
     * @param $name
     * @param $to_number
     * @param $from_number
     * @return BaseResponseInterface
     */
    public function sendUserInboundSMSReceivedNotification($user_id, $email, $name, $to_number, $from_number)
    {
        return $this->execute(__FUNCTION__, ['id' => $user_id], compact('email', 'name', 'to_number', 'from_number'));
    }

    /**
     * @param $user_id
     * @param $email
     * @param $subscription_id
     * @param $created_at
     * @param $number
     * @param $inbound_fee
     * @param $total_cost
     * @return BaseResponseInterface
     */
    public function sendUserInboundSMSReceiptNotification(
        $user_id,
        $email,
        $subscription_id,
        $created_at,
        $number,
        $inbound_fee,
        $total_cost
    )
    {
        return $this->execute(__FUNCTION__, ['id' => $user_id], compact('subscription_id', 'email', 'created_at', 'number', 'inbound_fee', 'total_cost'));
    }

    /**
     * @param $user_id
     * @param $email
     * @param $subscription_id
     * @param $created_at
     * @param $number
     * @param $inbound_fee
     * @param $total_cost
     * @return BaseResponseInterface
     */
    public function sendUserOptOutReceiptNotification(
        $user_id,
        $email,
        $subscription_id,
        $created_at,
        $number,
        $inbound_fee,
        $total_cost
    )
    {
        return $this->execute(__FUNCTION__, ['id' => $user_id], compact('subscription_id', 'email', 'created_at', 'number', 'inbound_fee', 'total_cost'));
    }

    /**
     * @return BaseResponseInterface
     */
    public function getAvailableNumbersForRenting()
    {
        return $this->execute(__FUNCTION__);
    }

    /**
     * @param $user_id
     * @param $payment_plan_id
     * @param $number
     * @param $tfn_usage
     * @param $tfn_opt_out_list_id
     * @return BaseResponseInterface
     */
    public function allocateNumber($user_id, $payment_plan_id, $number, $tfn_usage = null, $tfn_opt_out_list_id = null)
    {
        return $this->execute(__FUNCTION__, [], compact('user_id', 'payment_plan_id', 'number', 'tfn_usage', 'tfn_opt_out_list_id'));
    }
}
