<?php

namespace Intergo\HlrClient\UI;

use Intergo\BaseClient\Responses\BaseResponseInterface;

class LookupServiceUI
{
    /** @var string */
    public $baseUrl;

    /** @var LookupClient */
    protected $client;

    /** @var string */
    private $secret;

    /** @var string */
    protected $site;

    /** @var string */
    protected $traceId;

    /** @var string */
    protected $token;

    /**
     * LookupServiceUI constructor.
     */
    public function __construct($site, $baseUrl, $secret, $traceId, string $token = null)
    {
        $this->client = new LookupClient($baseUrl, $traceId);
        $this->secret = $secret;
        $this->site = $site;
        $this->token = $token;
    }

    /**
     * @param string $endpoint
     * @param $bodyParams
     * @param array $urlParams
     * @return BaseResponseInterface
     */
    protected function execute($endpoint, $bodyParams, array $urlParams = [])
    {
        return $this->client->execute($endpoint, $urlParams, $bodyParams);
    }
    
    private function prepareBodyData(array $data)
    {
        return array_merge(['token' => $this->token], $data);
    }

    ############################################### ESTIMATE ###############################################

    /**
     * Estimate the cost of performing number-lookup for the specified numbers.
     *
     * @param array|string $numbers
     * @param string $default_prefix
     * @return \stdClass
     * @throws \Exception
     */
    public function estimate($numbers, string $default_prefix = null)
    {
        return $this->execute(__FUNCTION__, $this->prepareBodyData(get_defined_vars()));
    }

    ############################################### LOOKUP ###############################################

    /**
     * Perform a number lookup.
     *
     * @param array|string $numbers
     * @param string $default_prefix
     * @param bool $is_api
     * @return \stdClass
     * @throws \Exception
     */
    public function lookup($numbers, string $default_prefix = null, bool $is_api = false)
    {
        return $this->execute(__FUNCTION__, $this->prepareBodyData(get_defined_vars()));
    }

    /**
     * Perform an instant number lookup.
     *
     * @param string $number
     * @param string $default_prefix
     * @return \stdClass
     * @throws \Exception
     */
    public function instantlookup($number, string $default_prefix = null)
    {
        return $this->execute('lookup', $this->prepareBodyData(get_defined_vars()));
    }

    ############################################### REPORTS ###############################################

    /**
     * Get the reports.
     *
     * string|array $queryString
     * @return BaseResponseInterface
     * @throws \Exception
     */
    public function getPaginatedReports($queryString)
    {
        if (is_string($queryString)) {
            parse_str($queryString, $queryString);
        }

        extract($queryString);
        unset($queryString);
        
        return $this->execute(__FUNCTION__, $this->prepareBodyData(get_defined_vars()));
    }

    /**
     * Initialize CSV export of the specified report.
     *
     * @param string $uuid
     * @return BaseResponseInterface
     * @throws \Exception
     */
    public function initializeExport(string $uuid)
    {
        return $this->execute(__FUNCTION__, $this->prepareBodyData([]), compact('uuid'));
    }

    /**
     * Get a listing of the exported files.
     *
     * @return BaseResponseInterface
     * @throws \Exception
     */
    public function getExports()
    {
        return $this->execute(__FUNCTION__, $this->prepareBodyData([]));
    }

    /**
     * Get a listing of the exported files.
     *
     * @param string $type
     * @param string $fileName
     * @return \stdClass
     * @throws \Exception
     */
    public function deleteExport(string $type, string $fileName)
    {
        $vars = get_defined_vars();
        return $this->execute(__FUNCTION__, $this->prepareBodyData([]),$vars);
    }

    ############################################### USER SETTINGS ###############################################

    /**
     * Get user-settings.
     *
     * For now, for this endpoint (/api/v2/user-settings) Lookup Service will only return user-settings for sending email
     * with export once when number-lookup is done, which is used on UI (where used can enable/disable this setting).
     * If you want to add/include more user-settings, please check code/app/Http/Resources/UserSettingsResource.php on
     * Lookup Service project.
     *
     * @return \stdClass
     * @throws \Exception
     */
    public function getUserSettings()
    {
        return $this->execute(__FUNCTION__, $this->prepareBodyData([]));
    }

    /**
     * Get user-settings.
     *
     * @param array $data
     * @return \stdClass
     * @throws \Exception
     */
    public function updateUserSettings(array $data)
    {
        extract($data);
        unset($data);

        return $this->execute(__FUNCTION__, $this->prepareBodyData(get_defined_vars()));
    }

    ############################################### GLOBAL SETTINGS ###############################################

    /**
     * Get global-settings.
     *
     * For now, for this endpoint (/api/v2/global-settings) Lookup Service will only return global-settings that is used
     * to determine if the number-lookup is enabled or disabled (if the user can request new lookups or not).
     * If you want to add/include more global-settings, please check code/app/Http/Resources/GlobalSettingsResource.php
     * on Lookup Service project.
     *
     * @return \stdClass
     * @throws \Exception
     */
    public function getGlobalSettings()
    {
        return $this->execute(__FUNCTION__, $this->prepareBodyData([]));
    }

    /**
     * Determine if the "Number-Lookup"is disabled.
     *
     * If it's disabled, then users will not be able to request new number-lookups.
     *
     * @return boolean
     * @throws \Exception
     */
    public function numberLookupIsDisabled()
    {
        $response = $this->getGlobalSettings();

        if ($response->isSuccessful()) {
            return $response->getData()['service_disabled'];
        }

        return true;
    }
}
