<?php

namespace Intergo\HlrClient\Manager;

use Intergo\BaseClient\Responses\BaseResponseInterface;

class LookupServiceManager
{
    /** @var LookupClient */
    protected $client;

    /** @var string */
    private $secret;

    /** @var string */
    protected $site;

    /** @var string */
    protected $token;

    /**
     * LookupServiceManager constructor.
     */
    public function __construct($site, $baseUrl, $secret, $traceId, string $token = null)
    {
        $this->client = new LookupClient($baseUrl, $traceId);
        $this->secret = $secret;
        $this->site = $site;
        $this->token = $token;
    }

    /**
     * @param string $endpoint
     * @param $bodyParams
     * @param array $urlParams
     * @return BaseResponseInterface
     */
    protected function execute($endpoint, $bodyParams, array $urlParams = [])
    {
        return $this->client->execute($endpoint, $urlParams, $bodyParams);
    }
    
    private function prepareBodyData(array $data)
    {
        return array_merge(['token' => $this->token], $data);
    }

    ############################################### REPORTS ###############################################

    /**
     * Get the paginated reports.
     *
     * @param string|array $queryString
     * @return LengthAwarePaginator
     * @throws \Exception
     */
    public function getPaginatedReports($queryString)
    {
        if (is_string($queryString)) {
            parse_str($queryString, $queryString);
        }

        extract($queryString);
        unset($queryString);

        return $this->execute(__FUNCTION__, $this->prepareBodyData(get_defined_vars()));
    }

    ############################################### NUMBER LOOKUPS ###############################################

    /**
     * Get the paginated reports.
     *
     * @param string|array $queryString
     * @return LengthAwarePaginator
     * @throws \Exception
     */
    public function getPaginatedNumberLookups($queryString)
    {
        if (is_string($queryString)) {
            parse_str($queryString, $queryString);
        }

        extract($queryString);
        unset($queryString);

        return $this->execute(__FUNCTION__, $this->prepareBodyData(get_defined_vars()));
    }

    /**
     * Get the stats for the specified user..
     *
     * @return array|object|string
     * @throws \Exception
     */
    public function getUserStats($user_id)
    {
        return $this->execute(__FUNCTION__, $this->prepareBodyData([]),compact('user_id'));
    }

    ############################################### PROVIDERS ###############################################

    /**
     * Get the paginated providers.
     *
     * @param string|array $queryString
     * @return LengthAwarePaginator
     * @throws \Exception
     */
    public function getPaginatedProviders($queryString)
    {
        if (is_string($queryString)) {
            parse_str($queryString, $queryString);
        }

        extract($queryString);
        unset($queryString);

        return $this->execute(__FUNCTION__, $this->prepareBodyData(get_defined_vars()));
    }

    /**
     * Get all providers.
     *
     * @return array|object|string
     * @throws \Exception
     */
    public function getAllProviders()
    {
        $page = 'no_pagination';
        return $this->execute(__FUNCTION__, $this->prepareBodyData([]));
    }

    /**
     * Get the provider by id.
     *
     * @param $id
     * @return array|object|string
     * @throws \Exception
     */
    public function getProviderById($id)
    {
        return $this->execute(__FUNCTION__, $this->prepareBodyData([]),compact('id'));
    }

    /**
     * Update the specified provider.
     *
     * @param $id
     * @param array $data
     * @return mixed
     * @throws \Exception
     */
    public function updateProvider($id, array $data)
    {
        extract($data);
        unset($data);

        return $this->execute(__FUNCTION__, $this->prepareBodyData(get_defined_vars()),compact('id'));
    }

    ############################################### PROVIDER-RATES ###############################################

    /**
     * Get the paginated provider-rates.
     *
     * @param string|array $queryString
     * @return LengthAwarePaginator
     * @throws \Exception
     */
    public function getPaginatedProviderRates($queryString)
    {
        if (is_string($queryString)) {
            parse_str($queryString, $queryString);
        }

        extract($queryString);
        unset($queryString);

        return $this->execute(__FUNCTION__, $this->prepareBodyData(get_defined_vars()));
    }

    /**
     * Get the provider-rate by id.
     *
     * @param $id
     * @return array|object|string
     * @throws \Exception
     */
    public function getProviderRateById($id)
    {
        return $this->execute(__FUNCTION__, $this->prepareBodyData([]),compact('id'));
    }

    /**
     * Create a new provider-rate.
     *
     * @param $id
     * @param array $data
     * @return mixed
     * @throws \Exception
     */
    public function storeProviderRate(array $data)
    {
        extract($data);
        unset($data);

        return $this->execute(__FUNCTION__, $this->prepareBodyData(get_defined_vars()));
    }

    /**
     * Update the specified provider-rate.
     *
     * @param $id
     * @param array $data
     * @return mixed
     * @throws \Exception
     */
    public function updateProviderRate($id, array $data)
    {
        extract($data);
        unset($data);

        return $this->execute(__FUNCTION__, $this->prepareBodyData(get_defined_vars()),compact('id'));
    }

    /**
     * Delete the specified provider-rate.
     *
     * @param $id
     * @return mixed
     * @throws \Exception
     */
    public function destroyProviderRate($id)
    {
        return $this->execute(__FUNCTION__, $this->prepareBodyData(get_defined_vars()),compact('id'));
    }

    /**
     * Import provider-rates.
     *
     * @param array $importRequest
     * @return mixed
     * @throws \Exception
     */
    public function importProviderRates(array $importRequest)
    {
        extract($importRequest);
        unset($importRequest);

        return $this->execute(__FUNCTION__, $this->prepareBodyData(get_defined_vars()));
    }

    ############################################### GLOBAL SETTINGS ############################################

    /**
     * Ge the global settings.
     *
     * @return array|object|string
     * @throws \Exception
     */
    public function getGlobalSettings()
    {
        return $this->execute(__FUNCTION__, $this->prepareBodyData(get_defined_vars()));
    }

    /**
     * Update global settings.
     *
     * @param array $data
     * @return array|object|string
     * @throws \Exception
     */
    public function updateGlobalSettings(array $data)
    {
        extract($data);
        unset($data);

        return $this->execute(__FUNCTION__, $this->prepareBodyData(get_defined_vars()));    
    }

    /**
     * Update 'lookup_disabled' global setting.
     *
     * This setting is updated separately from "/manager/settings/services-status" on APP
     *
     * @param array $data
     * @return array|object|string
     * @throws \Exception
     */
    public function updateLookupDisabled(array $data)
    {
        extract($data);
        unset($data);

        return $this->execute(__FUNCTION__, $this->prepareBodyData(get_defined_vars()));
    }

    ############################################### COUNTRY-RATES ############################################

    /**
     * Get the paginated country-rates.
     *
     * @param string|array $queryString
     * @return LengthAwarePaginator
     * @throws \Exception
     */
    public function getPaginatedCountryRates($queryString)
    {
        if (is_string($queryString)) {
            parse_str($queryString, $queryString);
        }

        extract($queryString);
        unset($queryString);

        return $this->execute(__FUNCTION__, $this->prepareBodyData(get_defined_vars()));
    }

    /**
     * Get all country-rates.
     *
     * @return array|object|string
     * @throws \Exception
     */
    public function getAllCountryRates()
    {
        $page = 'no_pagination';

        return $this->execute(__FUNCTION__, $this->prepareBodyData(get_defined_vars()));
    }

    /**
     * Get the country-rate by id.
     *
     * @param $id
     * @return array|object|string
     * @throws \Exception
     */
    public function getCountryRateById($id)
    {
        return $this->execute(__FUNCTION__, $this->prepareBodyData(get_defined_vars()),compact('id'));
    }

    /**
     * Create a new country-rate.
     *
     * @param array $data
     * @return mixed
     * @throws \Exception
     */
    public function storeCountryRate(array $data)
    {
        extract($data);
        unset($data);

        return $this->execute(__FUNCTION__, $this->prepareBodyData(get_defined_vars()));
    }

    /**
     * Update the specified country-rate.
     *
     * @param $id
     * @param array $data
     * @return mixed
     * @throws \Exception
     */
    public function updateCountryRate($id, array $data)
    {
        extract($data);
        unset($data);

        return $this->execute(__FUNCTION__, $this->prepareBodyData(get_defined_vars()),compact('id'));
    }

    /**
     * Delete the specified country-rate.
     *
     * @param $id
     * @return mixed
     * @throws \Exception
     */
    public function destroyCountryRate($id)
    {
        return $this->execute(__FUNCTION__, $this->prepareBodyData(get_defined_vars()),compact('id'));
    }

    ############################################### USER SETTINGS ############################################

    /**
     * Get the user settings.
     *
     * @param $id
     * @return array|object|string
     * @throws \Exception
     */
    public function getUserSettings($id)
    {
        return $this->execute(__FUNCTION__, $this->prepareBodyData(get_defined_vars()),compact('id'));
    }

    /**
     * Update user settings.
     *
     * @param $id
     * @param array $data
     * @return array|object|string
     * @throws \Exception
     */
    public function updateUserSettings($id, array $data)
    {
        extract($data);
        unset($data);

        return $this->execute(__FUNCTION__, $this->prepareBodyData(get_defined_vars()),compact('id'));
    }

    ############################################### ADJUSTED RATES ############################################

    /**
     * Get adjusted-rates.
     *
     * @param $user_id
     * @return array|object|string
     * @throws \Exception
     */
    public function getAdjustedRatesForUser($user_id)
    {
        return $this->execute(__FUNCTION__, $this->prepareBodyData(get_defined_vars()),compact('user_id'));
    }

    /**
     * Get the adjusted-rate by id.
     *
     * @param $id
     * @return array|object|string
     * @throws \Exception
     */
    public function getAdjustedRateById($id)
    {
        return $this->execute(__FUNCTION__, $this->prepareBodyData(get_defined_vars()),compact('id'));
    }

    /**
     * Create a new adjusted-rate.
     *
     * @param $user_id
     * @param array $data
     * @return mixed
     * @throws \Exception
     */
    public function storeAdjustedRate($user_id, array $data)
    {
        extract($data);
        unset($data);

        return $this->execute(__FUNCTION__, $this->prepareBodyData(get_defined_vars()),compact('user_id'));
    }

    /**
     * Update the specified adjusted-rate.
     *
     * @param $id
     * @param array $data
     * @return mixed
     * @throws \Exception
     */
    public function updateAdjustedRate($id, array $data)
    {
        extract($data);
        unset($data);

        return $this->execute(__FUNCTION__, $this->prepareBodyData(get_defined_vars()),compact('id'));
    }

    /**
     * Delete the specified adjusted-rate.
     *
     * @param $id
     * @return mixed
     * @throws \Exception
     */
    public function destroyAdjustedRate($id)
    {
        return $this->execute(__FUNCTION__, $this->prepareBodyData(get_defined_vars()),compact('id'));
    }

    ############################################### FAILED JOBS ############################################

    /**
     * Get the paginated failed jobs.
     *
     * @param string|array $queryString
     * @return LengthAwarePaginator
     * @throws \Exception
     */
    public function getPaginatedFailedJobs($queryString)
    {
        if (is_string($queryString)) {
            parse_str($queryString, $queryString);
        }

        extract($queryString);
        unset($queryString);

        return $this->execute(__FUNCTION__, $this->prepareBodyData(get_defined_vars()));
    }

    /**
     * Get failed queues.
     *
     * @return array|object|string
     * @throws \Exception
     */
    public function getFailedQueues()
    {
        return $this->execute(__FUNCTION__, $this->prepareBodyData(get_defined_vars()));
    }

    /**
     * Get the failed job by id.
     *
     * @param $id
     * @return array|object|string
     * @throws \Exception
     */
    public function getFailedJobById($id)
    {
        return $this->execute(__FUNCTION__, $this->prepareBodyData(get_defined_vars()),compact('id'));
    }

    /**
     * Rerun the specified failed job.
     *
     * @param $id
     * @return array|object|string
     * @throws \Exception
     */
    public function rerunFailedJob($id)
    {
        return $this->execute(__FUNCTION__, $this->prepareBodyData(get_defined_vars()),compact('id'));
    }

    /**
     * Delete the specified adjusted-rate.
     *
     * @param $id
     * @return mixed
     * @throws \Exception
     */
    public function destroyFailedJob($id)
    {
        return $this->execute(__FUNCTION__, $this->prepareBodyData(get_defined_vars()),compact('id'));
    }

    /**
     * Bulk rerun failed jobs.
     *
     * @param array $data
     * @return mixed
     * @throws \Exception
     */
    public function rerunBulk(array $data)
    {
        extract($data);
        unset($data);

        return $this->execute(__FUNCTION__, $this->prepareBodyData(get_defined_vars()));
    }

    /**
     * Bulk delete failed jobs.
     *
     * @param array $data
     * @return mixed
     * @throws \Exception
     */
    public function destroyBulk(array $data)
    {
        extract($data);
        unset($data);

        return $this->execute(__FUNCTION__, $this->prepareBodyData(get_defined_vars()));
    }
}
