<?php

namespace Intergo\Helpers;

use Intergo\Helpers\Calculators\SMSCounter;
use Intergo\Helpers\Exceptions\PhoneHelperInvalidPhoneNumberException;
use libphonenumber\NumberParseException;
use libphonenumber\PhoneNumber;
use libphonenumber\PhoneNumberFormat;
use libphonenumber\PhoneNumberToCarrierMapper;
use libphonenumber\PhoneNumberType;
use libphonenumber\PhoneNumberUtil;
use Intergo\Helpers\ValueObjects\PhoneNumberVO;
use Throwable;

class PhoneHelper
{
    /**
     * @param $phoneNumber
     * @param bool $sanitize
     * @return string
     */
    public static function cleanUpPhoneNumber($phoneNumber, bool $sanitize = false)
    {
        $phoneNumber = self::removeNonDigitCharacters($phoneNumber);
        if ($sanitize) {
            $phoneNumber = self::sanitizeToGSM($phoneNumber);
        }
        return '+' . self::removeLeadingZeros($phoneNumber);
    }


    /**
     * Prepend Plus sign on number if plus sign not exists already
     *
     * @param string $phoneNumber
     *
     * @return string
     */
    public static function prependPlusSign(string $phoneNumber)
    {
        return StringHelper::startsWith($phoneNumber, '+') ? $phoneNumber : '+' . $phoneNumber;
    }

    /**
     * Mask phone number.
     *
     * @param string $number
     * @return string
     */
    public static function maskPhoneNumber(string $number)
    {
        return substr($number, 0, 4) . '******' . substr($number, -2);
    }

    /**
     * @param $number
     * @return string
     */
    public static function removeNonDigitCharacters($number)
    {
        return preg_replace("/[^0-9]/", "", $number);
    }

    /**
     *
     * @param $number
     * @return string
     */
    public static function removeLeadingZeros($number)
    {
        return ltrim($number, '0');
    }

    /**
     * @param $number
     * @return array|string|string[]
     */
    private static function sanitizeToGSM($number)
    {
        $smsCounter = new SMSCounter();
        $number = $smsCounter->sanitizeToGSM($number);
        $number = preg_replace('/[\x00-\x08\x0B\x0C\x0E-\x1F\x7F-\x9F]/u', '', $number);
        $number = preg_replace('/[\x00-\x08\x0B\x0C\x0E-\x1F\x7F]/u', '', $number);
        $number = preg_replace('/[\x00-\x1F\x7F-\xA0\xAD]/u', '', $number);
        $number = preg_replace('/[^[:print:]]/', '', $number);
        return str_replace([' ', '-', '(', ')'], '', $number);
    }

    /**
     * @param string $phone
     * @param string|null $defaultPrefix
     * @param bool $returnBool
     * @return bool|PhoneNumberVO
     * @throws PhoneHelperInvalidPhoneNumberException
     */
    public static function validatePhoneNumber(string $phone, string $defaultPrefix = null, bool $returnBool = false)
    {
        $phoneNumber = null;
        $phoneNumberType = null;
        $phoneWithPlusSign = PhoneHelper::cleanUpPhoneNumber($phone);

        if(StringHelper::startsWith($phone,"+")){
            list($phoneNumber, $phoneNumberType) = self::validatePhoneNumberByGoogle($phoneWithPlusSign, null);
        }

        if (empty($phoneNumber)) {
            list($phoneNumber, $phoneNumberType) = self::validatePhoneNumberByGoogle(substr($phoneWithPlusSign, 1), strtoupper($defaultPrefix));
        }

        if (empty($phoneNumber)) {
            list($phoneNumber, $phoneNumberType) = self::validatePhoneNumberByGoogle($phoneWithPlusSign, null);
        }

        if ($returnBool) {
            return boolval($phoneNumber);
        }
        return self::getResult($phoneNumber, $phoneNumberType);
    }

    /**
     * Validate phone number using PhoneNumberUtil
     *
     * @param string $phone
     * @param string|null $defaultPrefix country code
     *
     * @return array
     */
    public static function validatePhoneNumberByGoogle($phone, ?string $defaultPrefix)
    {
        try {
            $phoneNumberUtil = PhoneNumberUtil::getInstance();
            $phoneNumber = $phoneNumberUtil->parse($phone, $defaultPrefix);
            $phoneNumberType = $phoneNumberUtil->getNumberType($phoneNumber);
            if (self::isValidMobileFormat($phoneNumberType)) {
                return [$phoneNumber, $phoneNumberType];
            }
        } catch (NumberParseException $exception) {
        }
        return [null, null];
    }

    /**
     * @param PhoneNumber|null $phoneNumber
     * @param int|null $phoneNumberType
     * @return PhoneNumberVO
     * @throws PhoneHelperInvalidPhoneNumberException
     */
    public static function getResult(?PhoneNumber $phoneNumber, ?int $phoneNumberType)
    {
        if (is_null($phoneNumber)) {
            throw new PhoneHelperInvalidPhoneNumberException('Invalid Number');
        }
        $phoneNumberUtil = PhoneNumberUtil::getInstance();
        $phone = $phoneNumberUtil->format($phoneNumber, PhoneNumberFormat::E164);
        $networkName = PhoneNumberToCarrierMapper::getInstance()->getNameForNumber($phoneNumber, "en");

        $phoneNumberVO = new PhoneNumberVO();
        $phoneNumberVO->setCountryCode($phoneNumber->getCountryCode());
        $phoneNumberVO->setNetworkName($networkName);
        $phoneNumberVO->setPhoneNumberAsString(self::prependPlusSign($phone));
        $phoneNumberVO->setPhoneNumber($phoneNumber);
        $phoneNumberVO->setA2Code($phoneNumberUtil->getRegionCodeForNumber($phoneNumber));
        $phoneNumberVO->setPhoneNumberType($phoneNumberType);
        return $phoneNumberVO;
    }

    /**
     * @param $phoneNumberType
     * @return bool
     */
    public static function isValidMobileFormat($phoneNumberType)
    {
        if (!in_array($phoneNumberType, self::getValidNumbers(), true)) {
            return false;
        }
        return true;
    }

    /**
     * @return int[]
     */
    private static function getValidNumbers(): array
    {
        return [
            PhoneNumberType::FIXED_LINE,
            PhoneNumberType::FIXED_LINE_OR_MOBILE,
            PhoneNumberType::MOBILE,
            PhoneNumberType::TOLL_FREE
        ];
    }

    /**
     * Format the number.
     *
     * If number is (has) decimal(s), then set 2 decimal points and add thousands-separator.
     * If it's a whole number, then only add thousands-separator.
     *
     * @param $number
     * @return string|mixed
     */
    public static function beautifyNumber($number)
    {
        try {
            if (is_numeric($number) && floor($number) != $number) {
                return number_format($number, 2, '.', ',');
            }
            return number_format($number, 0, '.', ',');
        } catch (Throwable $exception) {
            return $number;
        }
    }
}
