<?php

namespace Intergo\Helpers;

use DateTime;
use DateTimeZone;
use Exception;

class TimezoneHelper
{
    /**
     * @return array|false
     */
    private static function timezoneList()
    {
        return timezone_identifiers_list(DateTimeZone::ALL_WITH_BC);
    }

    /**
     * @return array
     */
    private static function timezones()
    {
        $timezones = [];
        $time = self::timezoneList();
        foreach ($time as $timezone) {
            $timezones[$timezone] = $timezone;
        }
        return $timezones;
    }

    /**
     * @return array
     */
    public static function getTimezones()
    {
        $timezones = DateTimeZone::listIdentifiers();
        $items = [];
        foreach ($timezones as $timezoneId) {
            $timezone = new DateTimeZone($timezoneId);
            $offsetInSeconds = $timezone->getOffset(new DateTime());
            $offsetPrefix = $offsetInSeconds < 0 ? '-' : '+';
            $offset = gmdate('H:i', abs($offsetInSeconds));
            $offset = "(GMT${offsetPrefix}${offset}) " . $timezoneId;
            $offsetInSeconds = $offset;
            $item = [
                'id' => $timezoneId,
                'name' => $offsetInSeconds
            ];
            $items[] = $item;
        }
        return $items;
    }

    /**
     * @param $timezone
     * @return bool
     */
    public static function isValidTimezone($timezone)
    {
        return isset(self::timezones()[$timezone]);
    }

    /**
     * @param $timezone
     * @return mixed
     */
    public static function getTimezoneName($timezone)
    {
        foreach (self::getTimezones() as $item) {
            if ($item['id'] == $timezone) {
                return $item['name'];
            }
        }
        return '';
    }

    /**
     * @param string $fromTime
     * @param string $fromTimezone
     * @param string $toTimezone
     * @return null
     */
    public static function convertTimezone(string $fromTime, string $fromTimezone = 'UTC', string $toTimezone = 'UTC')
    {
        if (ctype_digit($fromTime)) {
            $fromTime = date('Y-m-d H:i:s', $fromTime);
        }
        try {
            $orgTime = new DateTime($fromTime, new DateTimeZone($fromTimezone));
            $toTime = new DateTime($orgTime->format('c'));
            $toTime->setTimezone(new DateTimeZone($toTimezone));

            return $toTime->getTimestamp();
        } catch (Exception $ex) {
            return null;
        }
    }
}