<?php

namespace Intergo\Helpers;

use Intergo\Helpers\Calculators\SMSCounter;
use Intergo\Helpers\Exceptions\PhoneHelperInvalidPhoneNumberException;;
use Intergo\Helpers\ValueObjects\PhoneNumberVO;
use Throwable;

class PhoneHelper
{
    /**
     * @param $phoneNumber
     * @param bool $sanitize
     * @return string
     */
    public static function cleanUpPhoneNumber($phoneNumber, bool $sanitize = false)
    {
        $phoneNumber = self::removeNonDigitCharacters($phoneNumber);
        if ($sanitize) {
            $phoneNumber = self::sanitizeToGSM($phoneNumber);
        }
        return '+' . self::removeLeadingZeros($phoneNumber);
    }


    /**
     * Prepend Plus sign on number if plus sign not exists already
     *
     * @param string $phoneNumber
     *
     * @return string
     */
    public static function prependPlusSign(string $phoneNumber)
    {
        return StringHelper::startsWith($phoneNumber, '+') ? $phoneNumber : '+' . $phoneNumber;
    }

    /**
     * Mask phone number.
     *
     * @param string $number
     * @return string
     */
    public static function maskPhoneNumber(string $number)
    {
        return substr($number, 0, 4) . '******' . substr($number, -2);
    }

    /**
     * @param $number
     * @return string
     */
    public static function removeNonDigitCharacters($number)
    {
        return preg_replace("/[^0-9]/", "", $number);
    }

    /**
     *
     * @param $number
     * @return string
     */
    public static function removeLeadingZeros($number)
    {
        return ltrim($number, '0');
    }

    /**
     * @param $number
     * @return array|string|string[]
     */
    private static function sanitizeToGSM($number)
    {
        $smsCounter = new SMSCounter();
        $number = $smsCounter->sanitizeToGSM($number);
        $number = preg_replace('/[\x00-\x08\x0B\x0C\x0E-\x1F\x7F-\x9F]/u', '', $number);
        $number = preg_replace('/[\x00-\x08\x0B\x0C\x0E-\x1F\x7F]/u', '', $number);
        $number = preg_replace('/[\x00-\x1F\x7F-\xA0\xAD]/u', '', $number);
        $number = preg_replace('/[^[:print:]]/', '', $number);
        return str_replace([' ', '-', '(', ')'], '', $number);
    }

    /**
     * @param string $phone
     * @param string|null $defaultPrefix
     * @param bool $returnBool
     * @return bool|PhoneNumberVO
     * @throws PhoneHelperInvalidPhoneNumberException
     * @throws \Exception
     */
    public static function validatePhoneNumber(string $phone, string $defaultPrefix = null, bool $returnBool = false)
    {
        $url = rtrim('/', getenv('SERVICE_PHONE_VALIDATOR_URL'));
        if (empty($url)){
            throw new \Exception("Phone validator url is not set in .env file. Please set SERVICE_PHONE_VALIDATOR_URL in .env file.");
        }
        $url .= '/api/v1/validatePhoneNumber/' . $phone . '/' . $defaultPrefix;

        $curl = curl_init();

        curl_setopt_array($curl, array(
            CURLOPT_URL => $url,
            CURLOPT_RETURNTRANSFER => true,
            CURLOPT_FOLLOWLOCATION => true,
        ));

        $response = curl_exec($curl);
        if (curl_errno($curl)) {
            throw new \Exception(curl_error($curl));
        }

        curl_close($curl);
        $decodedResponse = json_decode($response, true);
        $isSuccess = $decodedResponse['success'];
        if ($returnBool) {
            return boolval($isSuccess);
        }

        if (!$isSuccess) {
            throw new PhoneHelperInvalidPhoneNumberException('Invalid Number');
        }
        $phoneNumberVO = new PhoneNumberVO();
        $phoneNumberVO->setCountryCode($decodedResponse['data']['country_code']);
        $phoneNumberVO->setNetworkName($decodedResponse['data']['network_name']);
        $phoneNumberVO->setPhoneNumber($decodedResponse['data']['phone_number_as_string']);
        $phoneNumberVO->setNationalNumber($decodedResponse['data']['phone_number']['national_number']);
        $phoneNumberVO->setA2Code($decodedResponse['data']['a2_code']);
        $phoneNumberVO->setPhoneNumberType($decodedResponse['data']['phone_number_type']);
        return $phoneNumberVO;
    }

    /**
     * Format the number.
     *
     * If number is (has) decimal(s), then set 2 decimal points and add thousands-separator.
     * If it's a whole number, then only add thousands-separator.
     *
     * @param $number
     * @return string|mixed
     */
    public static function beautifyNumber($number)
    {
        try {
            if (is_numeric($number) && floor($number) != $number) {
                return number_format($number, 2, '.', ',');
            }
            return number_format($number, 0, '.', ',');
        } catch (Throwable $exception) {
            return $number;
        }
    }
}
