<?php

namespace Intergo\Helpers\Calculators;

use stdClass;

class SMSCounter
{
    /**
     * GSM 7BIT encoding name.
     *
     * @var string
     */
    const GSM_7BIT = 'GSM_7BIT';

    /**
     * GSM 7BIT Extended encoding name.
     *
     * @var string
     */
    const GSM_7BIT_EX = 'GSM_7BIT_EX';

    /**
     * UTF16 or UNICODE encoding name.
     *
     * @var string
     */
    const UTF16 = 'UTF16';

    /**
     * Message length for GSM 7 Bit charset.
     *
     * @var int
     */
    const GSM_7BIT_LEN = 160;

    /**
     * Message length for GSM 7 Bit charset with extended characters.
     *
     * @var int
     */
    const GSM_7BIT_EX_LEN = 160;

    /**
     * Message length for UTF16/Unicode charset.
     *
     * @var int
     */
    const UTF16_LEN = 70;

    /**
     * Message length for multipart message in GSM 7 Bit encoding.
     *
     * @var int
     */
    const GSM_7BIT_LEN_MULTIPART = 153;

    /**
     * Message length for multipart message in GSM 7 Bit encoding.
     *
     * @var int
     */
    const GSM_7BIT_EX_LEN_MULTIPART = 153;

    /**
     * Message length for multipart message in GSM 7 Bit encoding.
     *
     * @var int
     */
    const UTF16_LEN_MULTIPART = 67;

    /**
     * Removes non GSM characters from a string.
     *
     * @param $str
     * @return string
     */
    public function removeNonGsmChars($str)
    {
        return $this->replaceNonGsmChars($str);
    }

    /**
     * Replaces non GSM characters from a string.
     *
     * @param string $str String to be replaced
     * @param string $replacement String of characters to be replaced with
     *
     * @return bool|string (string|false) if replacement string is more than 1 character
     *                        in length
     */
    public function replaceNonGsmChars($str, $replacement = null)
    {
        $validChars = $this->getGsm7bitExMap();
        $allChars = $this->utf8ToUnicode($str);
        if (strlen($replacement) > 1) {
            return false;
        }

        $replacementArray = [];
        $unicodeArray = $this->utf8ToUnicode($replacement);

        $replacementUnicode = array_pop($unicodeArray);
        foreach ($allChars as $key => $char) {
            if (!in_array($char, $validChars)) {
                $replacementArray[] = $key;
            }
        }

        if ($replacement) {
            foreach ($replacementArray as $key) {
                $allChars[$key] = $replacementUnicode;
            }
        }

        if (!$replacement) {
            foreach ($replacementArray as $key) {
                unset($allChars[$key]);
            }
        }

        return $this->unicodeToUtf8($allChars);
    }

    /**
     * @return int[]
     */
    public function getGsm7bitExMap()
    {
        return array_merge(
            $this->getGsm7bitMap(),
            $this->getAddedGsm7bitExMap()
        );
    }

    /**
     * @return int[]
     */
    public function getGsm7bitMap()
    {
        return [
            10,
            12,
            13,
            32,
            33,
            34,
            35,
            36,
            37,
            38,
            39,
            40,
            41,
            42,
            43,
            44,
            45,
            46,
            47,
            48,
            49,
            50,
            51,
            52,
            53,
            54,
            55,
            56,
            57,
            58,
            59,
            60,
            61,
            62,
            63,
            64,
            65,
            66,
            67,
            68,
            69,
            70,
            71,
            72,
            73,
            74,
            75,
            76,
            77,
            78,
            79,
            80,
            81,
            82,
            83,
            84,
            85,
            86,
            87,
            88,
            89,
            90,
            91,
            92,
            93,
            94,
            95,
            97,
            98,
            99,
            100,
            101,
            102,
            103,
            104,
            105,
            106,
            107,
            108,
            109,
            110,
            111,
            112,
            113,
            114,
            115,
            116,
            117,
            118,
            119,
            120,
            121,
            122,
            123,
            124,
            125,
            126,
            161,
            163,
            164,
            165,
            167,
            191,
            196,
            197,
            198,
            199,
            201,
            209,
            214,
            216,
            220,
            223,
            224,
            228,
            229,
            230,
            232,
            233,
            236,
            241,
            242,
            246,
            248,
            249,
            252,
            915,
            916,
            920,
            923,
            926,
            928,
            931,
            934,
            936,
            937,
            8364,
        ];
    }

    /**
     * @return int[]
     */
    public function getAddedGsm7bitExMap()
    {
        return [12, 91, 92, 93, 94, 123, 124, 125, 126, 8364];
    }

    /**
     * Generates array of unicode points for the utf8 string.
     *
     * @param $str
     * @return array
     */
    public function utf8ToUnicode($str)
    {
        $unicode = [];
        $values = [];
        $lookingFor = 1;
        $len = strlen($str);

        for ($i = 0; $i < $len; $i++) {
            $thisValue = ord($str[$i]);
            if ($thisValue < 128) {
                $unicode[] = $thisValue;
            }
            if ($thisValue >= 128) {
                if (count($values) === 0) {
                    $lookingFor = ($thisValue < 224) ? 2 : 3;
                }

                $values[] = $thisValue;

                if (count($values) === $lookingFor) {
                    $number = ($lookingFor === 3) ?
                        (($values[0] % 16) * 4096) + (($values[1] % 64) * 64) + ($values[2] % 64) :
                        (($values[0] % 32) * 64) + ($values[1] % 64);

                    $unicode[] = $number;
                    $values = [];
                    $lookingFor = 1;
                }
            }
        }

        return $unicode;
    }

    /**
     * Converts unicode code points array to a utf8 str.
     *
     * @param array $array unicode code-points array
     *
     * @return string utf8 encoded string
     */
    public function unicodeToUtf8($array)
    {
        $str = '';
        foreach ($array as $a) {
            $str .= $this->utf8Chr($a);
        }

        return $str;
    }

    /**
     * Converts unicode code points array to a utf8 str.
     *
     * @param array $array unicode code-points array
     *
     * @return array utf8 encoded array
     */
    public function unicodePosition($array)
    {
        $str = [];
        foreach ($array as $key => $a) {
            $str[$key] = $this->utf8Chr($a);
        }

        return $str;
    }

    /**
     * @param $unicode
     * @return string
     */
    public function utf8Chr($unicode)
    {
        $unicode = (int)$unicode;
        $utf8char = chr(240 | ($unicode >> 18));
        $utf8char .= chr(128 | (($unicode >> 12) & 0x3F));
        $utf8char .= chr(128 | (($unicode >> 6) & 0x3F));
        $utf8char .= chr(128 | ($unicode & 0x3F));
        if ($unicode < 128) {
            $utf8char = chr($unicode);
        } elseif ($unicode < 2048) {
            $utf8char = chr(192 | ($unicode >> 6)) . chr(128 | ($unicode & 0x3F));
        } elseif ($unicode < 65536) {
            $utf8char = chr(224 | ($unicode >> 12)) . chr(128 | (($unicode >> 6) & 0x3F)) . chr(
                    128 | ($unicode & 0x3F)
                );
        }

        return $utf8char;
    }

    /**
     * @param $str
     * @param array $customChars
     * @return string|string[]|null
     */
    public function sanitizeToGSM($str, array $customChars = [])
    {
        $str = $this->removeAccents($str, $customChars);
        return preg_replace('/\s+/', ' ', $str);
    }

    /**
     * Sanitize message to gsm for user
     *
     * @param string $str
     * @param bool $disableTranscode
     * @param array $customChars
     * @return string|string[]|null
     * @author Panayiotis Halouvas <phalouvas@kainotomo.com>
     */
    public function userSanitizeToGSM(string $str, bool $disableTranscode, $customChars = [])
    {
        if (!$disableTranscode) {
            $str = str_replace("\n", ":::", $str);
            $str = $this->sanitizeToGSM($str, $customChars);
            return str_replace(":::", "\n", $str);
        }
        return $str;
    }

    /**
     * @param string $str Message text
     *
     * @return string Sanitized message text
     */
    public function removeAccents($str, array $customChars = [])
    {
        $chars = array_merge($this->routeeChars(), $this->remainingChars(), $this->emptyChars(), $customChars);
        return strtr($str, $chars);
    }

    /**
     * @return string[]
     */
    protected function emptyChars()
    {
        return [
            "U+0000" => " ",
            " " => " ",
            " " => " ",
            "U+0003" => " ",
            "U+0004" => " ",
            "U+0009" => " ",
            "U+0010" => " ",
            "U+0011" => " ",
            "U+0012" => " ",
            "U+0013" => " ",
            "U+0014" => " ",
            "U+0017" => " ",
            "U+0019" => " ",
            "U+0080" => " ",
            "U+008D" => " ",
            "U+0090" => " ",
            "U+009B" => " ",
            "U+009F" => " ",
            "U+2000" => " ",
            "U+2001" => " ",
            "U+2002" => " ",
            "U+2003" => " ",
            "U+2004" => " ",
            "U+2005" => " ",
            "U+2006" => " ",
            "U+2007" => " ",
            "U+2008" => " ",
            "U+2009" => " ",
            "U+2028" => " ",
            "U+2029" => " ",
            "U+202F" => " ",
            "U+205F" => " ",
            "U+2060" => " ",
            "U+3000" => " ",
            "U+200A" => " ",
            "U+200B" => " ",
            "U+200C" => " ",
            "U+200D" => " ",
            "U+200E" => " ",
            "U+200F" => " ",
            " " => " ",
            " " => " ",
            " " => " ",
            " " => " ",
            " " => " ",
            " " => " ",
            " " => " ",
            " " => " ",
            "  " => " ",
            " " => " ",
            "　" => " ",
            "​" => " ",
            " " => " ",
            " " => " ",
            "👨🏻‍🚀" => " ",
            "‎" => " "
        ];
    }

    /**
     * @return string[]
     */
    protected function routeeChars()
    {
        return [
            '`' => "'",
            '¨' => '"',
            '«' => '"',
            '´' => "'",
            '»' => '"',
            'À' => 'A',
            'Á' => 'A',
            "Â" => "A",
            "È" => "E",
            "Ê" => "E",
            "Ë" => "E",
            "Ì" => "I",
            "Í" => "I",
            "Î" => "I",
            "Ï" => "I",
            "Ò" => "O",
            "Ó" => "O",
            "Ô" => "O",
            "Ù" => "U",
            "Ú" => "U",
            "Û" => "U",
            "à" => "à",
            "á" => "a",
            "â" => "a",
            "ê" => "e",
            "ë" => "e",
            "í" => "i",
            "î" => "i",
            "ï" => "i",
            "ó" => "o",
            "ô" => "o",
            "÷" => "/",
            "ú" => "u",
            "û" => "u",
            "ă" => "a",
            "ı" => "i",
            "Œ" => "Ö",
            "œ" => "ö",
            "Ş" => "S",
            "ǃ" => "!",
            "Ș" => "S",
            "ș" => "s",
            "Ț" => "T",
            "ț" => "t",
            "ɢ" => "G",
            "ɪ" => "I",
            "ɴ" => "N",
            "ʀ" => "R",
            "ʏ" => "Y",
            "ʙ" => "B",
            "ʜ" => "H",
            "ʟ" => "L",
            "ʹ" => "'",
            "ʺ" => "\"",
            "ʻ" => "'",
            "ʼ" => "'",
            "ʽ" => "'",
            "ˆ" => "^",
            "ˈ" => "'",
            "ˊ" => "'",
            "ˋ" => "'",
            "ː" => " => ",
            "˖" => "+",
            "˜" => "~",
            "ˮ" => "\"",
            "˷" => "~",
            "˸" => " => ",
            "̂" => "^",
            "̃" => "~",
            "̓" => "'",
            "̔" => "'",
            "̦" => ",",
            "̰" => "~",
            "̲" => "_",
            "̴" => "~",
            "̷" => "/",
            "̸" => "/",
            "͇" => "=",
            "Ά" => "A",
            "Έ" => "E",
            "Ή" => "H",
            "Ί" => "I",
            "Ό" => "O",
            "Ύ" => "Y",
            "Ώ" => "Ω",
            "ΐ" => "I",
            "Α" => "A",
            "Β" => "B",
            "Ε" => "E",
            "Ζ" => "Z",
            "Η" => "H",
            "Ι" => "I",
            "Κ" => "K",
            "Μ" => "M",
            "Ν" => "N",
            "Ο" => "O",
            "Ρ" => "P",
            "Τ" => "T",
            "Υ" => "Y",
            "Χ" => "X",
            "Ϊ" => "I",
            "Ϋ" => "Y",
            "ά" => "A",
            "έ" => "E",
            "ή" => "H",
            "ί" => "I",
            "ΰ" => "Y",
            "α" => "A",
            "β" => "B",
            "γ" => "Γ",
            "δ" => "Δ",
            "ε" => "E",
            "ζ" => "Z",
            "η" => "H",
            "θ" => "Θ",
            "ι" => "I",
            "κ" => "K",
            "λ" => "Λ",
            "μ" => "M",
            "ν" => "N",
            "ξ" => "Ξ",
            "ο" => "O",
            "π" => "Π",
            "ρ" => "P",
            "ς" => "Σ",
            "σ" => "Σ",
            "τ" => "T",
            "υ" => "Y",
            "φ" => "Φ",
            "χ" => "X",
            "ψ" => "Ψ",
            "ω" => "Ω",
            "ϊ" => "I",
            "ϋ" => "Y",
            "ό" => "O",
            "ύ" => "Y",
            "ώ" => "Ω",
            "ᴀ" => "A",
            "ᴄ" => "C",
            "ᴅ" => "D",
            "ᴇ" => "E",
            "ᴊ" => "J",
            "ᴋ" => "K",
            "ᴍ" => "M",
            "ᴏ" => "O",
            "ᴘ" => "P",
            "ᴛ" => "T",
            "ᴜ" => "U",
            "ᴠ" => "V",
            "ᴡ" => "W",
            "ᴢ" => "Z",
            "᷍" => "^",
            "‐" => "-",
            "–" => "-",
            "—" => "-",
            "―" => "-",
            "‗" => "_",
            "‘" => "'",
            "’" => "'",
            "‚" => ",",
            "‛" => "'",
            "“" => "\"",
            "”" => "\"",
            "„" => "\"",
            "‟" => "\"",
            "‹" => ">",
            "›" => "<",
            "⁃" => "-",
            "⁄" => "/",
            "⁎" => "*",
            "⁏" => ";",
            "⁠" => " ",
            "₤" => "£",
            "⃒" => "|",
            "⃓" => "|",
            "⃥" => "\\",
            "∕" => "/",
            "∗" => "*",
            "∣" => "|",
            "∼" => "~",
            "⊛" => "*",
            "⎜" => "|",
            "⎟" => "|",
            "⎸" => "|",
            "⎹" => "|",
            "⎼" => "-",
            "⎽" => "-",
            "⏐" => "|",
            "✢" => "*",
            "✣" => "*",
            "✤" => "*",
            "✥" => "*",
            "✱" => "*",
            "✲" => "*",
            "✳" => "*",
            "✺" => "*",
            "✻" => "*",
            "✼" => "*",
            "✽" => "*",
            "❃" => "*",
            "❉" => "*",
            "❊" => "*",
            "❋" => "*",
            "❛" => "'",
            "❜" => "'",
            "❝" => "\"",
            "❞" => "\"",
            "❨" => "(",
            "❩" => ")",
            "❪" => "(",
            "❫" => ")",
            "❴" => "{",
            "❵" => "}",
            "⟮" => "(",
            "⟯" => ")",
            "⦂" => " => ",
            "⦅" => "(",
            "⦆" => ")",
            "⧆" => "*",
            "⧵" => "\\",
            "⧸" => "/",
            "⧹" => "\\",
            "。" => ".",
            "〝" => "\"",
            "〞" => "\"",
            "ꜰ" => "F",
            "ꜱ" => "S",
            "꞉" => " => ",
            "꞊" => "=",
            "︐" => "'",
            "︑" => "'",
            "︓" => " => ",
            "︔" => ";",
            "︕" => "!",
            "︖" => "?",
            "﹐" => ",",
            "﹑" => ",",
            "﹒" => ".",
            "﹔" => ";",
            "﹖" => "?",
            "﹗" => "!",
            "﹙" => "(",
            "﹚" => ")",
            "﹛" => "{",
            "﹜" => "}",
            "﹟" => "#",
            "﹠" => "&",
            "﹡" => "*",
            "﹢" => "+",
            "﹣" => "-",
            "﹤" => "<",
            "﹥" => ">",
            "﹦" => "=",
            "﹨" => "\\",
            "﹩" => "$",
            "﹪" => "%",
            "﹫" => "@",
            "！" => "!",
            "＂" => "\"",
            "＃" => "#",
            "＄" => "$",
            "％" => "%",
            "＆" => "&",
            "＇" => "'",
            "（" => "(",
            "）" => ")",
            "＊" => "*",
            "＋" => "+",
            "，" => ",",
            "－" => "-",
            "．" => ".",
            "／" => "/",
            "０" => "0",
            "１" => "1",
            "２" => "2",
            "３" => "3",
            "４" => "4",
            "５" => "5",
            "６" => "6",
            "７" => "7",
            "８" => "8",
            "９" => "9",
            "：" => ":",
            "；" => ";",
            "＜" => "<",
            "＝" => "=",
            "＞" => ">",
            "？" => "?",
            "＠" => "@",
            "Ａ" => "A",
            "Ｂ" => "B",
            "Ｃ" => "C",
            "Ｄ" => "D",
            "Ｅ" => "E",
            "Ｆ" => "F",
            "Ｇ" => "G",
            "Ｈ" => "H",
            "Ｉ" => "I",
            "Ｊ" => "J",
            "Ｋ" => "K",
            "Ｌ" => "L",
            "Ｍ" => "M",
            "Ｎ" => "N",
            "Ｏ" => "O",
            "Ｐ" => "P",
            "Ｑ" => "Q",
            "Ｒ" => "R",
            "Ｓ" => "S",
            "Ｔ" => "T",
            "Ｕ" => "U",
            "Ｖ" => "V",
            "Ｗ" => "W",
            "Ｘ" => "X",
            "Ｙ" => "Y",
            "Ｚ" => "Z",
            "［" => "[",
            "＼" => "\\",
            "］" => "]",
            "＾" => "^",
            "＿" => "_",
            "｛" => "{",
            "｜" => "|",
            "｝" => "}",
            "～" => "~",
            "｡" => ".",
            "､" => ","
        ];
    }

    /**
     * @return string[]
     */
    protected function remainingChars()
    {
        return [
            "É" => "E",
            "ª" => "a",
            '΄' => "'",
            "º" => "o",
            "Ã" => "A",
            "Ð" => "D",
            "Õ" => "O",
            "Ý" => "Y",
            "Þ" => "TH",
            "ã" => "a",
            "ç" => "c",
            "ð" => "d",
            "õ" => "o",
            "ý" => "y",
            "þ" => "th",
            "ÿ" => "y",
            "Ā" => "A",
            "ā" => "a",
            "Ă" => "A",
            "Ą" => "A",
            "ą" => "a",
            "Ć" => "C",
            "ć" => "c",
            "Ĉ" => "C",
            "ĉ" => "c",
            "Ċ" => "C",
            "ċ" => "c",
            "Č" => "C",
            "č" => "c",
            "Ď" => "D",
            "ď" => "d",
            "Đ" => "D",
            "đ" => "d",
            "Ē" => "E",
            "ē" => "e",
            "Ĕ" => "E",
            "ĕ" => "e",
            "Ė" => "E",
            "ė" => "e",
            "Ę" => "E",
            "ę" => "e",
            "Ě" => "E",
            "ě" => "e",
            "Ĝ" => "G",
            "ĝ" => "g",
            "Ğ" => "G",
            "ğ" => "g",
            "Ġ" => "G",
            "ġ" => "g",
            "Ģ" => "G",
            "ģ" => "g",
            "Ĥ" => "H",
            "ĥ" => "h",
            "Ħ" => "H",
            "ħ" => "h",
            "Ĩ" => "I",
            "ĩ" => "i",
            "Ī" => "I",
            "ī" => "i",
            "Ĭ" => "I",
            "ĭ" => "i",
            "Į" => "I",
            "į" => "i",
            "İ" => "I",
            "Ĳ" => "IJ",
            "ĳ" => "ij",
            "Ĵ" => "J",
            "ĵ" => "j",
            "Ķ" => "K",
            "ķ" => "k",
            "ĸ" => "k",
            "Ĺ" => "L",
            "ĺ" => "l",
            "Ļ" => "L",
            "ļ" => "l",
            "Ľ" => "L",
            "ľ" => "l",
            "Ŀ" => "L",
            "ŀ" => "l",
            "Ł" => "L",
            "ł" => "l",
            "Ń" => "N",
            "ń" => "n",
            "Ņ" => "N",
            "ņ" => "n",
            "Ň" => "N",
            "ň" => "n",
            "ŉ" => "n",
            "Ŋ" => "N",
            "ŋ" => "n",
            "Ō" => "O",
            "ō" => "o",
            "Ŏ" => "O",
            "ŏ" => "o",
            "Ő" => "O",
            "ő" => "o",
            "Ŕ" => "R",
            "ŕ" => "r",
            "Ŗ" => "R",
            "ŗ" => "r",
            "Ř" => "R",
            "ř" => "r",
            "Ś" => "S",
            "ś" => "s",
            "Ŝ" => "S",
            "ŝ" => "s",
            "ş" => "s",
            "Š" => "S",
            "š" => "s",
            "Ţ" => "T",
            "ţ" => "t",
            "Ť" => "T",
            "ť" => "t",
            "Ŧ" => "T",
            "ŧ" => "t",
            "Ũ" => "U",
            "ũ" => "u",
            "Ū" => "U",
            "ū" => "u",
            "Ŭ" => "U",
            "ŭ" => "u",
            "Ů" => "U",
            "ů" => "u",
            "Ű" => "U",
            "ű" => "u",
            "Ų" => "U",
            "ų" => "u",
            "Ŵ" => "W",
            "ŵ" => "w",
            "Ŷ" => "Y",
            "ŷ" => "y",
            "Ÿ" => "Y",
            "Ź" => "Z",
            "ź" => "z",
            "Ż" => "Z",
            "ż" => "z",
            "Ž" => "Z",
            "ž" => "z",
            "ſ" => "s",
            "Ơ" => "O",
            "ơ" => "o",
            "Ư" => "U",
            "ư" => "u",
            "Ầ" => "A",
            "ầ" => "a",
            "Ằ" => "A",
            "ằ" => "a",
            "Ề" => "E",
            "ề" => "e",
            "Ồ" => "O",
            "ồ" => "o",
            "Ờ" => "O",
            "ờ" => "o",
            "Ừ" => "U",
            "ừ" => "u",
            "Ỳ" => "Y",
            "ỳ" => "y",
            "Ả" => "A",
            "ả" => "a",
            "Ẩ" => "A",
            "ẩ" => "a",
            "Ẳ" => "A",
            "ẳ" => "a",
            "Ẻ" => "E",
            "ẻ" => "e",
            "Ể" => "E",
            "ể" => "e",
            "Ỉ" => "I",
            "ỉ" => "i",
            "Ỏ" => "O",
            "ỏ" => "o",
            "Ổ" => "O",
            "ổ" => "o",
            "Ở" => "O",
            "ở" => "o",
            "Ủ" => "U",
            "ủ" => "u",
            "Ử" => "U",
            "ử" => "u",
            "Ỷ" => "Y",
            "ỷ" => "y",
            "Ẫ" => "A",
            "ẫ" => "a",
            "Ẵ" => "A",
            "ẵ" => "a",
            "Ẽ" => "E",
            "ẽ" => "e",
            "Ễ" => "E",
            "ễ" => "e",
            "Ỗ" => "O",
            "ỗ" => "o",
            "Ỡ" => "O",
            "ỡ" => "o",
            "Ữ" => "U",
            "ữ" => "u",
            "Ỹ" => "Y",
            "ỹ" => "y",
            "Ấ" => "A",
            "ấ" => "a",
            "Ắ" => "A",
            "ắ" => "a",
            "Ế" => "E",
            "ế" => "e",
            "Ố" => "O",
            "ố" => "o",
            "Ớ" => "O",
            "ớ" => "o",
            "Ứ" => "U",
            "ứ" => "u",
            "Ạ" => "A",
            "ạ" => "a",
            "Ậ" => "A",
            "ậ" => "a",
            "Ặ" => "A",
            "ặ" => "a",
            "Ẹ" => "E",
            "ẹ" => "e",
            "Ệ" => "E",
            "ệ" => "e",
            "Ị" => "I",
            "ị" => "i",
            "Ọ" => "O",
            "ọ" => "o",
            "Ộ" => "O",
            "ộ" => "o",
            "Ợ" => "O",
            "ợ" => "o",
            "Ụ" => "U",
            "ụ" => "u",
            "Ự" => "U",
            "ự" => "u",
            "Ỵ" => "Y",
            "ỵ" => "y",
            "ɑ" => "a",
            "Ǖ" => "U",
            "ǖ" => "u",
            "Ǘ" => "U",
            "ǘ" => "u",
            "Ǎ" => "A",
            "ǎ" => "a",
            "Ǐ" => "I",
            "ǐ" => "i",
            "Ǒ" => "O",
            "ǒ" => "o",
            "Ǔ" => "U",
            "ǔ" => "u",
            "Ǚ" => "U",
            "ǚ" => "u",
            "Ǜ" => "U",
            "ǜ" => "u"
        ];
    }

    /**
     * Truncated to the limit of chars allowed by number of SMS. It will detect
     * the encoding a multipart limits to apply the truncate.
     *
     * @param string $str Message text
     * @param $limitSms
     * @return string Truncated message
     */
    public function truncate($str, $limitSms)
    {
        $limit = 0;
        $count = $this->count($str);

        if ($count->messages <= $limitSms) {
            return $str;
        }

        if ($count->encoding === 'UTF16') {
            $limit = self::UTF16_LEN;

            if ($limitSms > 2) {
                $limit = self::UTF16_LEN_MULTIPART;
            }
        }

        if ($count->encoding !== 'UTF16') {
            $limit = self::GSM_7BIT_LEN;

            if ($limitSms > 2) {
                $limit = self::GSM_7BIT_LEN_MULTIPART;
            }
        }

        do {
            $str = mb_substr($str, 0, $limit * $limitSms);
            $count = $this->count($str);

            --$limit;
        } while ($count->messages > $limitSms);

        return $str;
    }

    /**
     * Detects the encoding, Counts the characters, message length, remaining characters.
     *
     * @param string $text
     * @param string $type
     * @return stdClass Object with params encoding,length, per_message, remaining, messages
     */
    public function count(string $text, string $type = 'sms')
    {
        if ($type == 'viber') {
            return $this->countViber($text);
        }
        return $this->countSms($text);
    }

    /**
     * Detects the encoding, Counts the characters, message length, remaining characters for SMS.
     *
     * @param string $text
     * @return stdClass Object with params encoding,length, per_message, remaining, messages
     */
    public function countViber(string $text)
    {
        $unicodeArray = $this->utf8ToUnicode($text);
        // variable to catch if any ex chars while encoding detection.
        $exChars = [];
        $encoding = $this->detectEncoding($unicodeArray, $exChars);
        $length = count($unicodeArray);

        if ($encoding === self::GSM_7BIT_EX) {
            $lengthExchars = count($exChars);
            // Each ex-char in the GSM 7 Bit encoding takes one more space
            // Hence the length increases by one char for each of those Ex chars.
            $length += $lengthExchars;
        }

        $perMessage = 1000;

        $messages = (int)ceil($length / $perMessage);
        $remaining = ($perMessage * $messages) - $length;

        $returnSet = new stdClass();

        $returnSet->encoding = $encoding;
        $returnSet->length = $length;
        $returnSet->per_message = $perMessage;
        $returnSet->remaining = $remaining;
        $returnSet->messages = $messages;

        return $returnSet;
    }

    /**
     * @param string $text
     * @return bool
     */
    public function isUnicode(string $text): bool
    {
        $unicodeArray = $this->utf8ToUnicode($text);
        $exChars = [];
        $encoding = $this->detectEncoding($unicodeArray, $exChars);

        switch ($encoding) {
            case self::GSM_7BIT_EX:
            case self::GSM_7BIT:
                return false;
            default:
                return true;
        }
    }

    /**
     * Detects the encoding, Counts the characters, message length, remaining characters for SMS.
     *
     * @param string $text
     * @return stdClass Object with params encoding,length, per_message, remaining, messages
     */
    public function countSms(string $text)
    {
        $unicodeArray = $this->utf8ToUnicode($text);
        // variable to catch if any ex chars while encoding detection.
        $exChars = [];
        $encoding = $this->detectEncoding($unicodeArray, $exChars);
        $length = count($unicodeArray);

        if ($encoding === self::GSM_7BIT_EX) {
            $lengthExchars = count($exChars);
            // Each ex-char in the GSM 7 Bit encoding takes one more space
            // Hence the length increases by one char for each of those Ex chars.
            $length += $lengthExchars;
        }

        // Select the per message length according to encoding and the message length
        switch ($encoding) {
            case self::GSM_7BIT:
                $perMessage = self::GSM_7BIT_LEN;
                if ($length > self::GSM_7BIT_LEN) {
                    $perMessage = self::GSM_7BIT_LEN_MULTIPART;
                }
                break;

            case self::GSM_7BIT_EX:
                $perMessage = self::GSM_7BIT_EX_LEN;
                if ($length > self::GSM_7BIT_EX_LEN) {
                    $perMessage = self::GSM_7BIT_EX_LEN_MULTIPART;
                }
                break;

            default:
                $perMessage = self::UTF16_LEN;
                if ($length > self::UTF16_LEN) {
                    $perMessage = self::UTF16_LEN_MULTIPART;
                }

                break;
        }

        $messages = (int)ceil($length / $perMessage);
        $remaining = ($perMessage * $messages) - $length;

        $returnSet = new stdClass();

        $returnSet->encoding = $encoding;
        $returnSet->length = $length;
        $returnSet->per_message = $perMessage;
        $returnSet->remaining = $remaining;
        $returnSet->messages = $messages;

        return $returnSet;
    }

    /**
     * @param $str
     * @param array $customChars
     * @return stdClass
     */
    public function detectUnicodePosition($str, array $customChars = [])
    {
        $unicodePositions = new stdClass();
        $unicodeArrayBeforeSanitization = $this->detectUnicode($str);
        $unicodePositions->detectedUnicode = $unicodeArrayBeforeSanitization;
        $unicodePositions->unknownUnicode = [];
        $str = $this->removeAccents($str, $customChars);
        $str = preg_replace('/\s+/', ' ', $str);
        $unicodeArrayAfterSanitization = $this->detectUnicode($str);
        if (count($unicodeArrayAfterSanitization)) {
            $unicodePositions->unknownUnicode = $unicodeArrayAfterSanitization;
        }
        return $unicodePositions;
    }

    /**
     * @param $text
     * @return array
     */
    public function detectUnicode($text)
    {
        if (!is_array($text)) {
            $text = $this->utf8ToUnicode($text);
        }

        $utf16Chars = array_diff($text, $this->getGsm7bitExMap());
        if (!count($utf16Chars)) {
            return [];
        }
        return $this->unicodePosition($utf16Chars);
    }

    /**
     * Detects the encoding of a particular text.
     *
     * @param $text
     * @param $exChars
     * @return string (GSM_7BIT|GSM_7BIT_EX|UTF16)
     */
    public function detectEncoding($text, &$exChars)
    {
        if (!is_array($text)) {
            $text = $this->utf8ToUnicode($text);
        }

        $utf16Chars = array_diff($text, $this->getGsm7bitExMap());

        if (count($utf16Chars)) {
            return self::UTF16;
        }

        $exChars = array_intersect($text, $this->getAddedGsm7bitExMap());

        if (count($exChars)) {
            return self::GSM_7BIT_EX;
        }

        return self::GSM_7BIT;
    }
}
