<?php

namespace Intergo\HashHelper;

use Throwable;

class HashHelper
{
    const DEFAULT_ALGO = 'sha256';

    /**
     * Generate hash based on different algorithm.
     * Use sha1 for NON-CRYPTOGRAPHIC info. Just to hash data to use it to match something. Not hashing passwords.
     * For password or other sensitive data use the self::DEFAULT_ALGO.
     *
     * @param $value
     * @param string $algo
     * @return string
     */
    public static function hash($value, string $algo = 'sha1')
    {
        try {
            return hash($algo, $value);
        } catch (Throwable $e) {
            return hash(self::DEFAULT_ALGO, $value);
        }
    }

    /**
     * Generate time based uuid version 1
     * by rearranging fields as discussed on https://mariadb.com/kb/en/guiduuid-performance/
     *
     * @param $user_id
     * @return string
     * @throws Throwable
     */
    public static function generateUniqueID($user_id)
    {
        $time = strtotime('now');
        $prefix = $user_id . "-" . substr($time, 0, 7); //13,14,15
        $random_string = bin2hex(random_bytes(20));
        $unique_id = self::hash(uniqid($random_string, true) . substr($time, 8), 'tiger128,4');
        $total_prefix_count = strlen($prefix);
        $append_part = "-"
            . substr($unique_id, 0, 4) . "-"
            . substr($unique_id, 4, 4) . "-"
            . substr($unique_id, 8, 8);
        $final_count = 35 - $total_prefix_count - strlen($append_part);
        if ($final_count > 0) {
            $append_part .= "-" . substr($unique_id, -$final_count);
        }
        return $prefix . $append_part;
    }

    /**
     * Mask (hide) the value that is in text (string).
     *
     * @param string $value
     * @param string $text
     * @param string $replaceWith
     * @return string
     */
    public static function maskValue($value, string $text, string $replaceWith = '*******')
    {
        if (!empty($value) && is_string($value)) {
            $value = urlencode($value);
            return str_replace($value, $replaceWith, $text);
        }
        return $text;
    }
}