<?php

namespace Intergo\CountryRegulationsClient;

use Intergo\BaseClient\Responses\BaseResponseInterface;

abstract class AbstractCountryRegulationsService
{
    protected CountryRegulationsClient $client;

    /**
     * AbstractSubscriptionService constructor.
     * https://intergo.atlassian.net/wiki/spaces/ES/pages/2149187587/Communication+between+microservices
     *
     * @param $baseUrl
     * @param $traceId
     */
    public function __construct($baseUrl, $traceId)
    {
        $this->client = new CountryRegulationsClient($baseUrl, $traceId);
    }

    /**
     * @param $attributes
     * @return  BaseResponseInterface
     */
    public function createCountryRegulation($attributes): BaseResponseInterface
    {
        extract($attributes);
        unset($attributes);
        return $this->execute(__FUNCTION__, get_defined_vars());
    }

    /**
     * Get country regulations
     * @param int $countryDialId The ID of the country dial
     * @return BaseResponseInterface
     */
    public function getCountryRegulations(int $countryDialId): BaseResponseInterface
    {
        return $this->execute(__FUNCTION__, [], compact("countryDialId"));
    }

    /**
     * Get country regulation by specific regulation id
     * @param int $id The ID of the country dial
     * @return BaseResponseInterface
     */
    public function getCountryRegulationById(int $id): BaseResponseInterface
    {
        return $this->execute(__FUNCTION__, [], compact("id"));
    }

    /**
     * @param string|null $regulationName
     * @return BaseResponseInterface
     */
    public function getCountryRegulationTypes(?string $regulationName = null): BaseResponseInterface
    {
        $queryParams = $regulationName ? ['name' => $regulationName] : [];
        return $this->execute(__FUNCTION__, $queryParams);
    }

    /**
     * Update specific country regulation
     * @param $id
     * @param $attributes
     * @return BaseResponseInterface
     */
    public function updateCountryRegulation($id,$attributes): BaseResponseInterface
    {
        extract($attributes);
        unset($attributes);
        return $this->execute(__FUNCTION__, get_defined_vars(),compact('id'));
    }

    /**
     * Delete specific country regulation based on the id of regulation
     * @param int $id
     * @return BaseResponseInterface
     */
    public function deleteCountryRegulation(int $id): BaseResponseInterface
    {
        return $this->execute(__FUNCTION__, [], compact('id'));
    }
    /**
     * Gets all the pre-defined registration requirements available
     * @return BaseResponseInterface
     */
    public function getRegistrationRequirements(): BaseResponseInterface
    {
        return $this->execute(__FUNCTION__, []);
    }
    /**
     * Get all available network providers
     * @return BaseResponseInterface
     */
    public function getNetworkProviders(): BaseResponseInterface
    {
        return $this->execute(__FUNCTION__, []);
    }



    /**
     * @param $endpoint
     * @param $bodyParams
     * @param array $urlParams
     * @return BaseResponseInterface
     */
    protected function execute($endpoint, $bodyParams, array $urlParams = []): BaseResponseInterface
    {
        $result = $this->client->execute($endpoint, $urlParams, $bodyParams);
        return $this->response($endpoint, $result);
    }

    abstract public function response(string $action, BaseResponseInterface $result);

}