<?php

namespace Intergo\BaseClient\Requests;

use Intergo\BaseClient\Enums\ContentTypes;
use GuzzleHttp\Exception\GuzzleException;
use Intergo\BaseClient\Exceptions\InvalidContentTypeHeaderException;
use Psr\Http\Message\ResponseInterface;

class BaseRequest implements RequestInterface
{
    /** @var */
    protected $url;

    /** @var RequestHandler */
    protected $http;
    /**
     * @var string
     */
    protected $trace_id;

    /**
     * BaseRequest constructor.
     * @param string $url
     * @param string $trace_id
     * @param array $config
     */
    public function __construct(string $url, string $trace_id = '1', array $config = [])
    {
        $this->url = $url;
        $this->trace_id = $trace_id;
        $this->http = new RequestHandler($config);
    }

    /**
     * @param $method
     * @param $options
     * @return mixed|ResponseInterface
     * @throws GuzzleException
     * @throws InvalidContentTypeHeaderException
     */
    public function execute($method, $options)
    {
        $options = array_replace_recursive($this->authorizationsHeaders(), $options);
        $response = $this->send($method, $options);
        if (isset($options['headers']['Content-Type']) && $options['headers']['Content-Type'] !== ContentTypes::APPLICATION_JSON) {
            throw new InvalidContentTypeHeaderException("Invalid Header: Content-Type must be application/json");
        }
        return json_decode($response->getBody()->getContents(), true);
    }

    /**
     * @param $method
     * @param array $params
     * @return ResponseInterface
     * @throws GuzzleException
     */
    public function send($method, array $params = [])
    {
        return $this->http->request($method, $this->url, $params);
    }

    /**
     * @return array
     */
    public function authorizationsHeaders()
    {
        return [
            'headers' => [
                'Accept' => ContentTypes::APPLICATION_JSON,
                'Content-Type' => ContentTypes::APPLICATION_JSON,
                'X-TRACE-ID' => $this->trace_id
            ]
        ];
    }
}
