<?php

namespace Intergo\BaseClient;

use Intergo\BaseClient\Enums\HttpMethodEnum;

abstract class ApiHandler
{
    /**
     * Classes that extend ApiHandler should have a list of $endpoints
     * with the following structure:
     *
     * protected static $endpoints = [
     *    'login' => [self::METHOD => HttpMethodEnum::POST, self::URL => '/auth/login'],
     *    'example => [self::METHOD => HttpMethodEnum::GET, self::URL => '/example/:userId'],
     *     ...
     */

    protected static $endpoints = [];
    protected static $apiPrefix = '/api';
    protected static $apiVersion = '/v1';

    const METHOD = 'method';
    const URL = 'url';

    /**
     * @param $endpoint
     * @param $params
     * @return string
     */
    public static function getUrl($endpoint, $params)
    {
        $url = static::$endpoints[$endpoint][self::URL] ?? '/';
        if (!empty($params)) {
            $url = self::prepareParameterizedURL($url, $params);
        }
        return sprintf("%s%s%s", static::$apiPrefix, static::$apiVersion, $url);
    }

    /**
     * @param $endpoint
     * @return string
     */
    public static function getMethod($endpoint)
    {
        return static::$endpoints[$endpoint][self::METHOD] ?? HttpMethodEnum::POST;
    }

    /**
     * @param string $url
     * @param array $params
     * @return string
     */
    protected static function prepareParameterizedURL(string $url, array $params): string
    {
        $parameters = [];
        foreach($params as $field => $value) {
            $parameters[':' . $field] = $value;
        }
        return strtr($url, $parameters);
    }
}
