<?php

namespace Intergo\AuthClient;

use Intergo\BaseClient\Enums\HttpMethodEnum;
use Intergo\BaseClient\Responses\BaseResponseInterface;

abstract class AbstractAuthService
{
    private AuthClient $client;
    protected string $secret;
    protected string $site;

    const LOGIN_RESULT_TOKEN = 'token';
    const LOGIN_RESULT_USER = 'user';


    /**
     * AbstractAuthService constructor.
     * https://intergo.atlassian.net/wiki/spaces/ES/pages/2149187587/Communication+between+microservices
     *
     * @param $site
     * @param $baseUrl
     * @param $secret
     * @param $traceId
     */
    public function __construct($site, $baseUrl, $secret, $traceId)
    {
        $this->client = new AuthClient($baseUrl, $traceId);
        $this->secret = $secret;
        $this->site = $site;
    }

    /**
     * @param $params
     * @return array
     */
    protected function setParams($params)
    {
        return array_merge([
            'site' => $this->site,
            'jwt_auth_secret' => $this->secret,
        ], $params);
    }

    /**
     * @param $endpoint
     * @param $bodyParams
     * @param array $urlParams
     * @return BaseResponseInterface
     */
    protected function execute($endpoint, $bodyParams, array $urlParams = [])
    {
        $bodyParams = $this->setParams($bodyParams);
        $email = $bodyParams['email'] ?? $bodyParams['username'] ?? null;
        $result = $this->client->execute($endpoint, $urlParams, $bodyParams);
        return $this->response($endpoint, $bodyParams['ip'] ?? '', $result, $email);
    }

    /**
     * @param $endpoint
     * @param $bearerToken
     * @param $bodyParams
     * @param array $urlParams
     * @return BaseResponseInterface
     */
    protected function executeAuth($endpoint, $bearerToken, $bodyParams, array $urlParams = [])
    {
        $bodyParams = $this->setParams($bodyParams);
        $email = $bodyParams['email'] ?? $bodyParams['username'] ?? null;
        $result = $this->client->executeAuth($endpoint, $bearerToken, $urlParams, $bodyParams);
        return $this->response($endpoint, $bodyParams['ip'] ?? '', $result, $email);
    }

    /**
     * @param string $action
     * @param $ip
     * @param BaseResponseInterface $result
     * @param ?string $email
     * @return BaseResponseInterface
     */
    abstract public function response(string $action, $ip, BaseResponseInterface $result, ?string $email = null);

    abstract public function storeJWT($authToken, $authUser);


    /**
     * AUTH ACTIONS
     */

    /**
     * @param $attributes
     * @return BaseResponseInterface
     */
    public function register($attributes)
    {
        return $this->execute(__FUNCTION__, $attributes);
    }

    /**
     * @param $username
     * @param $password
     * @param $ip
     * @return BaseResponseInterface
     */
    public function login($username, $password, $ip)
    {
        $result = $this->execute(__FUNCTION__, get_defined_vars());
        if ($result->isSuccessful()) {
            $this->storeJWT($result->getData()[self::LOGIN_RESULT_TOKEN], $result->getData()[self::LOGIN_RESULT_USER]);
        }
        return $result;
    }

    /**
     * @param string $token
     * @param $ip
     * @param bool $bypass_rules
     * @return BaseResponseInterface
     */
    public function authenticateWeb(string $token, $ip, bool $bypass_rules = false)
    {
        $result = $this->execute(__FUNCTION__, get_defined_vars());
        if ($result->isSuccessful()) {
            $this->storeJWT($token, $result->getData());
            return $this->response('Authenticate', $ip, $result);
        }
        if ($result->getMessage() == "Token has expired") {
            $result = $this->refresh($token, $ip);
            if ($result->isSuccessful()) {
                $token = $result->getData()[self::LOGIN_RESULT_TOKEN];
                $this->storeJWT($token, $result->getData()[self::LOGIN_RESULT_USER]);
                $result = $this->execute(__FUNCTION__, compact('token', 'ip', 'bypass_rules'));
            }
        }
        return $result;
    }

    /**
     * @param string $token
     * @param $ip
     * @param bool $bypass_rules
     * @return BaseResponseInterface
     */
    public function authenticateApi(string $token, $ip, bool $bypass_rules = false)
    {
        return $this->response('Authenticate', $ip, $this->execute(__FUNCTION__, get_defined_vars()));
    }

    /**
     * @param string $token
     * @param $ip
     * @param bool $bypass_rules
     * @return BaseResponseInterface
     */
    public function authenticateServiceWeb(string $token, $ip, bool $bypass_rules = true)
    {
        return $this->response('Authenticate', $ip, $this->execute('authenticateWeb', get_defined_vars()));
    }

    /**
     * @param string $token
     * @param $ip
     * @return BaseResponseInterface
     */
    public function refresh(string $token, $ip)
    {
        $result = $this->execute(__FUNCTION__, get_defined_vars());
        if ($result->isSuccessful()) {
            $this->storeJWT($result->getData()[self::LOGIN_RESULT_TOKEN], $result->getData()[self::LOGIN_RESULT_USER]);
        }
        return $result;
    }

    /**
     * @param string $token
     * @param $ip
     * @return BaseResponseInterface
     */
    public function invalidate(string $token, $ip)
    {
        return $this->execute(__FUNCTION__, get_defined_vars());
    }

    /**
     * @param $ip
     * @param $email
     * @return BaseResponseInterface
     */
    public function verifyEmail($ip, $email)
    {
        return $this->execute(__FUNCTION__, get_defined_vars());
    }

    /**
     * @param array $user
     * @param array $attributes
     * @param $ip
     */
    public function invalidateUserModel(array $user, array $attributes, $ip)
    {
        $attempts = 3;
        do {
            $attempts--;
            $result = $this->execute(__FUNCTION__, get_defined_vars());
        } while (!$result->isSuccessful() && $attempts > 0);
    }

    /**
     * @param string|null $email
     * @param $ip
     * @param bool $bypass_rules
     * @param bool $storeToken
     * @return BaseResponseInterface
     */
    public function loginBy(string $email, $ip, bool $storeToken = true, bool $bypass_rules = false)
    {
        $result = $this->execute(__FUNCTION__, compact('email','ip', 'bypass_rules'));
        if ($result->isSuccessful() && $storeToken) {
            $this->storeJWT($result->getData()[self::LOGIN_RESULT_TOKEN], $result->getData()[self::LOGIN_RESULT_USER]);
        }
        return $result;
    }

    /**
     * @param $payload
     * @return BaseResponseInterface
     */
    public function oAuthToken($payload)
    {
        $endpoint = '/oauth/token';
        $result = $this->client->executeCustom($endpoint, HttpMethodEnum::POST, [], $payload);
        return $this->response($endpoint, '', $result);
    }

    /**
     * API KEYS
     */

    /**
     * @param $email
     * @param $ip
     * @param $token
     * @return BaseResponseInterface
     */
    public function getUserApiKeys($email, $ip, $token)
    {
        return $this->execute(__FUNCTION__, get_defined_vars());
    }

    /**
     * @param $user_id
     * @return BaseResponseInterface
     */
    public function getApiKeysByUserId($user_id)
    {
        return $this->execute(__FUNCTION__, [], compact('user_id'));
    }

    /**
     * @param string $title
     * @param string $token
     * @param string $ip
     * @param string $email
     * @param int|null $user_id
     * @param bool $secure
     * @param array $allowed_ips
     * @return BaseResponseInterface
     */
    public function generateApiKey(
        string $title,
        string $token,
        string $ip,
        string $email,
        int $user_id = null,
        bool $secure = false,
        array $allowed_ips = []
    ): BaseResponseInterface {
        return $this->execute(__FUNCTION__, get_defined_vars());
    }

    /**
     * @param string $title
     * @param string $api_key
     * @param int $api_id
     * @param string $token
     * @param string $ip
     * @param string $email
     * @param array $allowed_ips
     * @return BaseResponseInterface
     */
    public function disableApiKey(
        string $title,
        string $api_key,
        int $api_id,
        string $token,
        string $ip,
        string $email,
        array $allowed_ips
    ) {
        return $this->execute(__FUNCTION__, get_defined_vars());
    }

    /**
     * @param string $title
     * @param string $api_key
     * @param int $api_id
     * @param string $token
     * @param string $ip
     * @param string $email
     * @param array $allowed_ips
     * @return BaseResponseInterface
     */
    public function enableApiKey(
        string $title,
        string $api_key,
        int $api_id,
        string $token,
        string $ip,
        string $email,
        array $allowed_ips
    ) {
        return $this->execute(__FUNCTION__, get_defined_vars());
    }

    /**
     * @param string $api_key
     * @param string $token
     * @param string $ip
     * @param string $email
     * @param int $api_id
     * @return BaseResponseInterface
     */
    public function deleteApiKey(string $api_key, string $token, string $ip, string $email, int $api_id)
    {
        return $this->execute(__FUNCTION__, get_defined_vars());
    }

    /**
     * @param int $user_id
     * @param string $ip
     * @param string $email
     * @return BaseResponseInterface
     */
    public function deleteUserApiKeys(int $user_id, string $ip, string $email)
    {
        return $this->execute(__FUNCTION__, get_defined_vars());
    }

    /**
     * @param int $user_id
     * @return BaseResponseInterface
     */
    public function getUserWebApiKey(int $user_id)
    {
        return $this->execute(__FUNCTION__, [], compact('user_id'));
    }


    /**
     * API CLIENTS
     */

    /**
     * @param string $token
     * @param string $ip
     * @param string $email
     * @return BaseResponseInterface
     */
    public function getUserApiClients(string $token, string $ip, string $email)
    {
        return $this->execute(__FUNCTION__, get_defined_vars());
    }

    /**
     * @param string $title
     * @param string $token
     * @param string $ip
     * @param string $email
     * @return BaseResponseInterface
     */
    public function generateApiClient(string $title, string $token, string $ip, string $email)
    {
        return $this->execute(__FUNCTION__, get_defined_vars());
    }

    /**
     * @param string $title
     * @param string $client_id
     * @param int $api_id
     * @param string $token
     * @param string $ip
     * @param string $email
     * @return BaseResponseInterface
     */
    public function disableApiClient(string $title, string $client_id, int $api_id, string $token, string $ip, string $email)
    {
        return $this->execute(__FUNCTION__, get_defined_vars());
    }

    /**
     * @param string $title
     * @param string $client_id
     * @param int $api_id
     * @param string $token
     * @param string $ip
     * @param string $email
     * @return BaseResponseInterface
     */
    public function enableApiClient(string $title, string $client_id, int $api_id, string $token, string $ip, string $email)
    {
        return $this->execute(__FUNCTION__, get_defined_vars());
    }

    /**
     * @param string $api_key
     * @param string $token
     * @param string $ip
     * @param string $email
     * @param int $api_id
     * @return BaseResponseInterface
     */
    public function deleteApiClient(string $api_key, string $token, string $ip, string $email, int $api_id)
    {
        return $this->execute(__FUNCTION__, get_defined_vars());
    }

    /**
     * Users Actions
     */

    /**
     * @param string $ip
     * @param string $email
     * @param string $password
     * @return BaseResponseInterface
     */
    public function resetPassword(string $ip, string $email, string $password)
    {
        $result = $this->execute(__FUNCTION__, get_defined_vars());
        if ($result->isSuccessful()) {
            $this->storeJWT($result->getData()[self::LOGIN_RESULT_TOKEN], $result->getData()[self::LOGIN_RESULT_USER]);
        }
        return $result;
    }

    /**
     * @param string $token
     * @param string $ip
     * @param string $email
     * @param string $password
     * @param string $newpassword
     * @return BaseResponseInterface
     */
    public function changePassword(string $token, string $ip, string $email, string $password, string $newpassword)
    {
        return $this->execute(__FUNCTION__, get_defined_vars());
    }

    /**
     * @param $user_id
     * @param $ip
     * @return BaseResponseInterface
     */
    public function getUserDetail($user_id, $ip): BaseResponseInterface
    {
        return $this->execute(__FUNCTION__, compact('ip'),compact('user_id'));
    }

    /**
     * @param $ids
     * @return BaseResponseInterface
     */
    public function getUsersDetails($ids): BaseResponseInterface
    {
        return $this->execute(__FUNCTION__, compact('ids'));
    }

    /**
     * @return BaseResponseInterface
     */
    public function getUsersQueues(): BaseResponseInterface
    {
        return $this->execute(__FUNCTION__, []);
    }

    /**
     * @param $search
     * @param $ip
     * @return BaseResponseInterface
     */
    public function searchUser($search, $ip): BaseResponseInterface
    {
        return $this->execute(__FUNCTION__, compact('search','ip'));
    }

    /**
     * @param string $token
     * @param string $ip
     * @param string $email
     * @param array $params
     * @return BaseResponseInterface
     */
    public function updateUser(string $token, string $ip, string $email, $params)
    {
        $parameters = array_merge(compact('token', 'ip','email'), $params);
        return $this->execute(__FUNCTION__, $parameters);
    }

    /**
     * @param string $token
     * @param string $ip
     * @param string $email
     * @return BaseResponseInterface
     */
    public function removeSendingRestriction(string $token, string $ip, string $email)
    {
        return $this->execute(__FUNCTION__, get_defined_vars());
    }

    /**
     * @param $bearerToken
     * @return BaseResponseInterface
     */
    public function getUserLegacyBalance($bearerToken): BaseResponseInterface
    {
        return $this->executeAuth(__FUNCTION__, $bearerToken, []);
    }

    /**
     * Roles & Permissions Actions
     */

    /**
     * @param $token
     * @param $ip
     * @param $email
     * @param $user_id
     * @return BaseResponseInterface
     */
    public function hasHigherRoleThan($token, $ip, $email, $user_id)
    {
        return $this->execute(__FUNCTION__, compact('token', 'email', 'ip'), compact('user_id'));
    }

    /**
     * @param $token
     * @param $ip
     * @param $email
     * @param $user_id
     * @return BaseResponseInterface
     */
    public function getRoles($token, $ip, $email, $user_id)
    {
        return $this->execute(__FUNCTION__, get_defined_vars());
    }

    /**
     * @param $token
     * @param $ip
     * @param $email
     * @param $role_id
     * @return BaseResponseInterface
     */
    public function getPermissions($token, $ip, $email, $role_id)
    {
        return $this->execute(__FUNCTION__, get_defined_vars());
    }

    /**
     * @param $token
     * @param $ip
     * @param $email
     * @param $name
     * @param $description
     * @param array $permissions
     * @return BaseResponseInterface
     */
    public function storeRole($token, $ip, $email, $name, $description, array $permissions = [])
    {
        return $this->execute(__FUNCTION__, get_defined_vars());
    }

    /**
     * @param $token
     * @param $ip
     * @param $email
     * @param $role_id
     * @param $name
     * @param $description
     * @param array $permissions
     * @return BaseResponseInterface
     */
    public function updateRole($token, $ip, $email, $role_id, $name, $description, array $permissions = [])
    {
        return $this->execute(__FUNCTION__, get_defined_vars());
    }

    /**
     * @param $token
     * @param $ip
     * @param $email
     * @param $role_id
     * @return BaseResponseInterface
     */
    public function deleteRole($token, $ip, $email, $role_id)
    {
        return $this->execute(__FUNCTION__, get_defined_vars());
    }

    /**
     * @param $token
     * @param $ip
     * @param $email
     * @param $user_id
     * @return BaseResponseInterface
     */
    public function getRolesForUser($token, $ip, $email, $user_id)
    {
        return $this->execute(__FUNCTION__, get_defined_vars());
    }

    /**
     * @param $token
     * @param $ip
     * @param $email
     * @param $user_id
     * @return BaseResponseInterface
     */
    public function getPermissionsForUser($token, $ip, $email, $user_id)
    {
        return $this->execute(__FUNCTION__, get_defined_vars());
    }

    /**
     * @param $token
     * @param $ip
     * @param $email
     * @return BaseResponseInterface
     */
    public function getUsersRolesAndPermissions($token, $ip, $email)
    {
        return $this->execute(__FUNCTION__, get_defined_vars());
    }

    /**
     * @param $token
     * @param $ip
     * @param $email
     * @param $query
     * @return BaseResponseInterface
     */
    public function searchForUser($token, $ip, $email, $query)
    {
        return $this->execute(__FUNCTION__, get_defined_vars());
    }

    /**
     * @param $token
     * @param $ip
     * @param $email
     * @param $query
     * @return BaseResponseInterface
     */
    public function searchForManagers($token, $ip, $email, $query)
    {
        return $this->execute(__FUNCTION__, get_defined_vars());
    }

    /**
     * @param $token
     * @param $ip
     * @param $email
     * @param $user_id
     * @param array $roles
     * @param array $permissions
     * @return BaseResponseInterface
     */
    public function assignOrRevokeRoles($token, $ip, $email, $user_id, array $roles = [], array $permissions = [])
    {
        return $this->execute(__FUNCTION__, get_defined_vars());
    }

    /**
     * Banlists & Whitelists Actions
     */

    /**
     * @param $token
     * @param $ip
     * @param $email
     * @param $match_string
     * @param $page
     * @return BaseResponseInterface
     */
    public function getEmailBanlist($token, $ip, $email, $match_string, $page)
    {
        return $this->execute(__FUNCTION__, get_defined_vars());
    }

    /**
     * @param $token
     * @param $ip
     * @param $email
     * @param $emails
     * @return BaseResponseInterface
     */
    public function addEmailBanlist($token, $ip, $email, $emails)
    {
        return $this->execute(__FUNCTION__, get_defined_vars());
    }

    /**
     * @param $token
     * @param $ip
     * @param $email
     * @param $emails
     * @return BaseResponseInterface
     */
    public function editEmailBanlist($token, $ip, $email, $emails)
    {
        return $this->execute(__FUNCTION__, get_defined_vars());
    }

    /**
     * @param $token
     * @param $ip
     * @param $email
     * @param $emails
     * @return BaseResponseInterface
     */
    public function removeEmailBanlist($token, $ip, $email, $emails)
    {
        return $this->execute(__FUNCTION__, get_defined_vars());
    }

    /**
     * @param $token
     * @param $ip
     * @param $email
     * @param $match_string
     * @param $page
     * @return BaseResponseInterface
     */
    public function getIpBanlist($token, $ip, $email, $match_string, $page)
    {
        return $this->execute(__FUNCTION__, get_defined_vars());
    }

    /**
     * @param $token
     * @param $ip
     * @param $email
     * @param $ip_addresses
     * @return BaseResponseInterface
     */
    public function addIpBanlist($token, $ip, $email, $ip_addresses)
    {
        return $this->execute(__FUNCTION__, get_defined_vars());
    }

    /**
     * @param $token
     * @param $ip
     * @param $email
     * @param $ip_addresses
     * @return BaseResponseInterface
     */
    public function editIpBanlist($token, $ip, $email, $ip_addresses)
    {
        return $this->execute(__FUNCTION__, get_defined_vars());
    }

    /**
     * @param $token
     * @param $ip
     * @param $email
     * @param $ip_addresses
     * @return BaseResponseInterface
     */
    public function removeIpBanlist($token, $ip, $email, $ip_addresses)
    {
        return $this->execute(__FUNCTION__, get_defined_vars());
    }

    /**
     * @param $token
     * @param $ip
     * @param $email
     * @param $match_string
     * @param $page
     * @return BaseResponseInterface
     */
    public function getEmailWhitelist($token, $ip, $email, $match_string, $page)
    {
        return $this->execute(__FUNCTION__, get_defined_vars());
    }

    /**
     * @param $token
     * @param $ip
     * @param $email
     * @param $emails
     * @return BaseResponseInterface
     */
    public function addEmailWhitelist($token, $ip, $email, $emails)
    {
        return $this->execute(__FUNCTION__, get_defined_vars());
    }

    /**
     * @param $token
     * @param $ip
     * @param $email
     * @param $emails
     * @return BaseResponseInterface
     */
    public function editEmailWhitelist($token, $ip, $email, $emails)
    {
        return $this->execute(__FUNCTION__, get_defined_vars());
    }

    /**
     * @param $token
     * @param $ip
     * @param $email
     * @param $emails
     * @return BaseResponseInterface
     */
    public function removeEmailWhitelist($token, $ip, $email, $emails)
    {
        return $this->execute(__FUNCTION__, get_defined_vars());
    }

    /**
     * @param $token
     * @param $ip
     * @param $email
     * @param $match_string
     * @param $page
     * @return BaseResponseInterface
     */
    public function getIpWhitelist($token, $ip, $email, $match_string, $page)
    {
        return $this->execute(__FUNCTION__, get_defined_vars());
    }

    /**
     * @param $token
     * @param $ip
     * @param $email
     * @param $ip_addresses
     * @return BaseResponseInterface
     */
    public function addIpWhitelist($token, $ip, $email, $ip_addresses)
    {
        return $this->execute(__FUNCTION__, get_defined_vars());
    }

    /**
     * @param $token
     * @param $ip
     * @param $email
     * @param $ip_addresses
     * @return BaseResponseInterface
     */
    public function editIpWhitelist($token, $ip, $email, $ip_addresses)
    {
        return $this->execute(__FUNCTION__, get_defined_vars());
    }

    /**
     * @param $token
     * @param $ip
     * @param $email
     * @param $ip_addresses
     * @return BaseResponseInterface
     */
    public function removeIpWhitelist($token, $ip, $email, $ip_addresses)
    {
        return $this->execute(__FUNCTION__, get_defined_vars());
    }

    /**
     * @param $token
     * @param $ip
     * @param $email
     * @param $match_string
     * @param $page
     * @return BaseResponseInterface
     */
    public function getEmailStopforum($token, $ip, $email, $match_string, $page)
    {
        return $this->execute(__FUNCTION__, get_defined_vars());
    }

    /**
     * @param $token
     * @param $ip
     * @param $email
     * @param $domain
     * @param $page
     * @return BaseResponseInterface
     */
    public function getFreeEmailBanlist($token, $ip, $email, $domain, $page)
    {
        return $this->execute(__FUNCTION__, get_defined_vars());
    }

    /**
     * @param $token
     * @param $ip
     * @param $email
     * @param $domains
     * @return BaseResponseInterface
     */
    public function addFreeEmailBanlist($token, $ip, $email, $domains)
    {
        return $this->execute(__FUNCTION__, get_defined_vars());
    }

    /**
     * @param $token
     * @param $ip
     * @param $email
     * @param $domains
     * @return BaseResponseInterface
     */
    public function editFreeEmailBanlist($token, $ip, $email, $domains)
    {
        return $this->execute(__FUNCTION__, get_defined_vars());
    }

    /**
     * @param $token
     * @param $ip
     * @param $email
     * @param $domains
     * @return BaseResponseInterface
     */
    public function removeFreeEmailBanlist($token, $ip, $email, $domains)
    {
        return $this->execute(__FUNCTION__, get_defined_vars());
    }

    /**
     * Admin Actions
     */

    /**
     * Approve the specified user.
     *
     * @param $token
     * @param $email
     * @param $ip
     * @param $userId
     * @param bool $via_twofa
     * @return BaseResponseInterface
     */
    public function approveUser($token, $email, $ip, $userId, bool $via_twofa = false)
    {
        return $this->execute(__FUNCTION__, get_defined_vars());
    }

    /**
     * Disapprove the specified user.
     *
     * @param $token
     * @param $email
     * @param $ip
     * @param $userId
     * @return BaseResponseInterface
     */
    public function disapproveUser($token, $email, $ip, $userId)
    {
        return $this->execute(__FUNCTION__, get_defined_vars());
    }

    /**
     * Ban the specified user.
     *
     * @param $token
     * @param $email
     * @param $ip
     * @param $user_id
     * @param string $ban_reason
     * @param bool $quarantine
     * @return BaseResponseInterface
     */
    public function banUser($token, $email, $ip, $user_id, string $ban_reason, bool $quarantine)
    {
        return $this->execute(__FUNCTION__, compact('token', 'email', 'ip', 'ban_reason', 'quarantine'), compact('user_id'));
    }

    /**
     * Unban the specified user.
     *
     * @param $token
     * @param $email
     * @param $ip
     * @param $user_id
     * @param bool $whitelisted
     * @return BaseResponseInterface
     */
    public function unbanUser($token, $email, $ip, $user_id, bool $whitelisted)
    {
        return $this->execute(__FUNCTION__, compact('token', 'email', 'ip','whitelisted'), compact('user_id'));
    }

    /**
     * Unban the specified user.
     *
     * @param $token
     * @param $email
     * @param $ip
     * @param $user_id
     * @return BaseResponseInterface
     */
    public function restoreUser($token, $email, $ip, $user_id)
    {
        return $this->execute(__FUNCTION__, compact('token', 'email', 'ip'), compact('user_id'));
    }

    /**
     * Assign service to the specified user.
     *
     * @param $token
     * @param $email
     * @param $ip
     * @param $user_id
     * @param $service_name
     * @return BaseResponseInterface
     */
    public function assignService($token, $email, $ip, $user_id, $service_name)
    {
        return $this->execute(__FUNCTION__, compact('token', 'email', 'ip'), compact('user_id', 'service_name'));
    }

    /**
     * Revoke service from the specified user.
     *
     * @param $token
     * @param $email
     * @param $ip
     * @param $user_id
     * @param $service_name
     * @return BaseResponseInterface
     */
    public function revokeService($token, $email, $ip, $user_id, $service_name)
    {
        return $this->execute(__FUNCTION__, compact('token', 'email', 'ip'), compact('user_id', 'service_name'));
    }


    /**
     * @param string $token
     * @param string $ip
     * @param string $email
     * @param string $user_id
     * @param array $params
     * @return BaseResponseInterface
     */
    public function updateUserByAdmin(string $token, string $ip, string $email, string $user_id, $params)
    {
        $parameters = array_merge(compact('token', 'ip', 'email'), $params);
        return $this->execute(__FUNCTION__, $parameters, compact('user_id'));
    }

    /**
     * Update sending restrictions (if the user can send only to own verified phone number).
     *
     * @param $token
     * @param $email
     * @param $ip
     * @param $user_id
     * @param bool $restricted_to_own_number
     * @return BaseResponseInterface
     */
    public function updateSendingRestriction($token, $email, $ip, $user_id, $restricted_to_own_number)
    {
        return $this->execute(__FUNCTION__, compact('token', 'email', 'ip','restricted_to_own_number'), compact('user_id'));
    }

    /**
     * Get activity stats
     *
     * @param $token
     * @param $email
     * @param $ip
     * @param string $start
     * @param string $end
     * @param string $interval
     * @return BaseResponseInterface
     */
    public function getActivityStats($token, $email, $ip, string $start, string $end, string $interval)
    {
        return $this->execute(__FUNCTION__, get_defined_vars());
    }

    /**
     * @param string $token
     * @param string $ip
     * @param string $email
     * @param string $user_id
     * @param $password
     * @return BaseResponseInterface
     */
    public function resetPasswordByAdmin(string $token, string $ip, string $email, string $user_id, $password)
    {
        return $this->execute(__FUNCTION__, compact('token', 'email', 'ip', 'password'), compact('user_id'));
    }

    /**
     * Rules Actions
     */

    /**
     *
     * @param $email
     * @param $ip
     * @param $user_id
     * @return BaseResponseInterface
     */
    public function validateRules($email = null, $ip = null, $user_id = null)
    {
        return $this->execute(__FUNCTION__, get_defined_vars());
    }

    /**
     *
     * @param $ip
     * @param $user_id
     * @return BaseResponseInterface
     */
    public function validateUserstatus($ip = null, $user_id = null)
    {
        return $this->execute(__FUNCTION__, get_defined_vars());
    }

    /**
     *
     * @param $ip
     * @param $ip_address
     * @return BaseResponseInterface
     */
    public function validateStopforumip($ip = null, $ip_address)
    {
        return $this->execute(__FUNCTION__, get_defined_vars());
    }

    /**
     *
     * @param $email
     * @param $ip
     * @return BaseResponseInterface
     */
    public function validateStopforumtoxicemail($email = null, $ip = null)
    {
        return $this->execute(__FUNCTION__, get_defined_vars());
    }

    /**
     *
     * @param $ip
     * @param $ip_address
     * @return BaseResponseInterface
     */
    public function validateDuplicateip($ip = null, $ip_address)
    {
        return $this->execute(__FUNCTION__, get_defined_vars());
    }

    /**
     *
     * @param $email
     * @param $ip
     * @return BaseResponseInterface
     */
    public function validateEmailbanlist($email = null, $ip = null)
    {
        return $this->execute(__FUNCTION__, get_defined_vars());
    }

    /**
     *
     * @param $ip
     * @param $ip_address
     * @return BaseResponseInterface
     */
    public function validateIpbanlist($ip = null, $ip_address)
    {
        return $this->execute(__FUNCTION__, get_defined_vars());
    }

    /**
     *
     * @param $email
     * @param $ip
     * @return BaseResponseInterface
     */
    public function validateEmailvalidation($email = null, $ip = null)
    {
        return $this->execute(__FUNCTION__, get_defined_vars());
    }

    /**
     *
     * @param $email
     * @param $ip
     * @return BaseResponseInterface
     */
    public function validateDuplicategmail($email = null, $ip = null)
    {
        return $this->execute(__FUNCTION__, get_defined_vars());
    }

    /**
     *
     * @param $email
     * @param $ip
     * @return BaseResponseInterface
     */
    public function validateDisposableemaildomain($email = null, $ip = null)
    {
        return $this->execute(__FUNCTION__, get_defined_vars());
    }

    /**
     *
     * @param $email
     * @param $ip
     * @return BaseResponseInterface
     */
    public function validateFreeemaildomain($email = null, $ip = null)
    {
        return $this->execute(__FUNCTION__, get_defined_vars());
    }

    /**
     * Clear IP rate event.
     *
     * @param string $token
     * @param string $ip
     * @param string $email
     * @param string $ip_address
     * @return BaseResponseInterface
     */
    public function clearIPRateEvent(string $token, string $ip, string $email, string $ip_address)
    {
        return $this->execute(__FUNCTION__, get_defined_vars());
    }

    /**
     * Webhook Secrets
     */

    /**
     * @param $user_id
     * @param $ip
     * @return BaseResponseInterface
     */
    public function getWebhookSecret($user_id, $ip): BaseResponseInterface
    {
        return $this->execute(__FUNCTION__, compact('ip'), compact('user_id'));
    }


    /**
     * @param $user_id
     * @param $ip
     * @param $token
     * @return BaseResponseInterface
     */
    public function getUserWebhookSecrets($user_id, $ip, $token)
    {
        return $this->execute(__FUNCTION__, get_defined_vars());
    }


    /**
     * @param $token
     * @param $ip
     * @param $email
     * @param $id
     * @param $secret
     * @param array $permissions
     * @return BaseResponseInterface
     */
    public function updateUserWebhookSecret($id, $secret, $token, $ip, $email,  array $permissions = [])
    {
        return $this->execute(__FUNCTION__, get_defined_vars());
    }

    /**
     * 
     *  MICROSERVICE ACTIONS
     */

    /**
     * Balance Actions
     */

    /**
     * @param $user_id
     * @param $amount
     * @param $note
     * @param $ip
     * @return BaseResponseInterface
     */
    public function addBalance($user_id, $amount, $note, $ip, $reason = ""): BaseResponseInterface
    {
        return $this->execute(__FUNCTION__, compact('amount','note','ip', 'reason'),compact('user_id'));
    }

    /**
     * @param $user_id
     * @param $amount
     * @param $note
     * @param $ip
     * @param string $reason
     * @return BaseResponseInterface
     */
    public function deductBalance($user_id, $amount, $note, $ip, $reason = ""): BaseResponseInterface
    {
        return $this->execute(__FUNCTION__, compact('amount','note','ip', 'reason'),compact('user_id'));
    }
}