<?php

namespace Intergo\AuthClient;

use Intergo\BaseClient\Responses\BaseResponseInterface;

abstract class AbstractAuthService
{
    /** @var string */
    public $baseUrl;
    /** @var AuthClient */
    private $client;
    /** @var string */
    private $secret;
    /** @var string */
    protected $site;
    /** @var string */
    protected $traceId;

    const LOGIN_RESULT_TOKEN = 'token';
    const LOGIN_RESULT_USER = 'user';

    /**
     * AuthService constructor.
     */
    public function __construct($site, $baseUrl, $secret, $traceId)
    {
        $this->baseUrl = $baseUrl;
        $this->client = new AuthClient();
        $this->secret = $secret;
        $this->site = $site;
        $this->traceId = $traceId;
    }

    /**
     * @param $params
     * @return array
     */
    private function setParams($params)
    {
        return array_merge([
            'site' => $this->site,
            'jwt_auth_secret' => $this->secret,
        ], $params);
    }

    /**
     * @param $endpoint
     * @param $bodyParams
     * @param array $urlParams
     * @return BaseResponseInterface
     */
    private function execute($endpoint, $bodyParams, $urlParams = [])
    {
        $bodyParams = $this->setParams($bodyParams);
        $result = $this->client->execute($endpoint, $this->baseUrl, $urlParams, $bodyParams, $this->traceId);
        return $this->response($endpoint, $params['email'] ?? '', $params['ip'] ?? '', $result);
    }

    /**
     * @param string $action
     * @param $email
     * @param $ip
     * @param BaseResponseInterface $result
     * @return BaseResponseInterface
     */
    abstract public function response(string $action, $email, $ip, BaseResponseInterface $result);

    abstract public function storeJWT($authToken, $authUser);


    /** ################################################ AUTH ACTIONS ################################################ */

    /**
     * @param $attributes
     * @return BaseResponseInterface
     */
    public function register($attributes)
    {
        return $this->execute(__FUNCTION__, get_defined_vars());
    }

    /**
     * @param $username
     * @param $password
     * @param $ip
     * @return mixed
     */
    public function login($username, $password, $ip)
    {
        $result = $this->execute(__FUNCTION__, get_defined_vars());
        if ($result->isSuccessful()) {
            $this->storeJWT($result->getData()[self::LOGIN_RESULT_TOKEN], $result->getData()[self::LOGIN_RESULT_USER]);
        }
        return $result;
    }

    /**
     * @param string $token
     * @param $ip
     * @param $email
     * @return array
     */
    public function verifyUser(string $token, $ip, $email)
    {
        $result = $this->execute(__FUNCTION__, get_defined_vars());
        if ($result->isSuccessful()) {
            return $result->getData();
        }
        return [
            'success' => false,
            'message' => $result->getMessage(),
            'data' => $result->getData()
        ];
    }

    /**
     * @param string $token
     * @param $ip
     * @param $email
     * @return BaseResponseInterface
     */
    public function authenticate(string $token, $ip, $email)
    {
        $result = $this->execute(__FUNCTION__, get_defined_vars());
        if ($result->isSuccessful()) {
            $this->storeJWT(null, $result->getData());
            return $this->response('Authenticate', $email, $ip, $result);
        }
        if ($result->getMessage() == "Token has expired") {
            $result = $this->refresh($token,$ip, $email);
            if ($result->isSuccessful()) {
                $token = $result->getData()[self::LOGIN_RESULT_TOKEN];
                $this->storeJWT($token, $result->getData()[self::LOGIN_RESULT_USER]);
                $result = $this->execute(__FUNCTION__, compact('token', 'ip', 'email'));
            }
        }
        return $result;
    }

    /**
     * @param string $token
     * @param $ip
     * @param $email
     * @return BaseResponseInterface
     */
    public function refresh(string $token, $ip, $email)
    {
        $result = $this->execute(__FUNCTION__, get_defined_vars());
        if ($result->isSuccessful()) {
            $this->storeJWT($result->getData()[self::LOGIN_RESULT_TOKEN], $result->getData()[self::LOGIN_RESULT_USER]);
        }
        return $result;
    }

    /**
     * @param string $token
     * @param $ip
     * @param $email
     * @return BaseResponseInterface
     */
    public function invalidate(string $token, $ip, $email)
    {
        return $this->execute(__FUNCTION__, get_defined_vars());
    }

    /**
     * @param string $confirm_code
     * @param $ip
     * @param $email
     * @return BaseResponseInterface
     */
    public function verifyEmail(string $confirm_code, $ip, $email)
    {
        return $this->execute(__FUNCTION__, get_defined_vars());
    }

    /**
     * @param array $user
     * @param array $attributes
     */
    public function invalidateUserModel(array $user, array $attributes)
    {
        $attempts = 3;
        do {
            $attempts--;
            $result = $this->execute(__FUNCTION__, get_defined_vars());
        } while (!$result->isSuccessful() && $attempts > 0);
    }

    /**
     * @param string|null $email
     * @param $ip
     * @return BaseResponseInterface
     */
    public function loginBy(string $email, $ip)
    {
        $result = $this->execute(__FUNCTION__, get_defined_vars());
        if ($result->isSuccessful()) {
            $this->storeJWT($result->getData()[self::LOGIN_RESULT_TOKEN], $result->getData()[self::LOGIN_RESULT_USER]);
        }
        return $result;
    }

    /** ################################################ API KEYS ################################################ */

    /**
     * @param $email
     * @param $ip
     * @param $token
     * @return BaseResponseInterface
     */
    public function getUserApiKeys($email, $ip, $token)
    {
        return $this->execute(__FUNCTION__, get_defined_vars());
    }

    /**
     * @param string $title
     * @param string $token
     * @param string $ip
     * @param string $email
     * @param int|null $user_id
     * @param bool $secure
     * @return BaseResponseInterface
     */
    public function generateApiKey(string $title, string $token, string $ip, string $email, int $user_id = null, bool $secure = false)
    {
        return $this->execute(__FUNCTION__, get_defined_vars());
    }

    /**
     * @param string $title
     * @param string $api_key
     * @param int $id
     * @param string $token
     * @param string $ip
     * @param string $email
     * @return BaseResponseInterface
     */
    public function disableApiKey(string $title, string $api_key, int $id, string $token, string $ip, string $email)
    {
        return $this->execute(__FUNCTION__, get_defined_vars());
    }

    /**
     * @param string $title
     * @param string $api_key
     * @param int $id
     * @param string $token
     * @param string $ip
     * @param string $email
     * @return BaseResponseInterface
     */
    public function enableApiKey(string $title, string $api_key, int $id, string $token, string $ip, string $email)
    {
        return $this->execute(__FUNCTION__, get_defined_vars());
    }

    /**
     * @param string $api_key
     * @param string $token
     * @param string $ip
     * @param string $email
     * @param int $id
     * @return BaseResponseInterface
     */
    public function deleteApiKey(string $api_key, string $token, string $ip, string $email, int $id)
    {
        return $this->execute(__FUNCTION__, get_defined_vars());
    }

    /**
     * @param int $user_id
     * @param string $ip
     * @param string $email
     * @return BaseResponseInterface
     */
    public function deleteUserApiKeys(int $user_id, string $ip, string $email)
    {
        return $this->execute(__FUNCTION__, get_defined_vars());
    }


    /** ################################################ API CLIENTS ################################################ */

    /**
     * @param string $token
     * @param string $ip
     * @param string $email
     * @return BaseResponseInterface
     */
    public function getUserApiClients(string $token, string $ip, string $email)
    {
        return $this->execute(__FUNCTION__, get_defined_vars());
    }

    /**
     * @param string $title
     * @param string $token
     * @param string $ip
     * @param string $email
     * @return BaseResponseInterface
     */
    public function generateApiClient(string $title, string $token, string $ip, string $email)
    {
        return $this->execute(__FUNCTION__, get_defined_vars());
    }

    /**
     * @param string $title
     * @param string $client_id
     * @param int $id
     * @param string $token
     * @param string $ip
     * @param string $email
     * @return BaseResponseInterface
     */
    public function disableApiClient(string $title, string $client_id, int $id, string $token, string $ip, string $email)
    {
        return $this->execute(__FUNCTION__, get_defined_vars());
    }

    /**
     * @param string $title
     * @param string $client_id
     * @param int $id
     * @param string $token
     * @param string $ip
     * @param string $email
     * @return BaseResponseInterface
     */
    public function enableApiClient(string $title, string $client_id, int $id, string $token, string $ip, string $email)
    {
        return $this->execute(__FUNCTION__, get_defined_vars());
    }

    /**
     * @param string $api_key
     * @param string $token
     * @param string $ip
     * @param string $email
     * @param int $id
     * @return BaseResponseInterface
     */
    public function deleteApiClient(string $api_key, string $token, string $ip, string $email, int $id)
    {
        return $this->execute(__FUNCTION__, get_defined_vars());
    }

    /** ################################################ Roles & Permissions Actions ################################################ */

    /**
     * @param $token
     * @param $ip
     * @param $email
     * @param $user_id
     * @return BaseResponseInterface
     */
    public function hasHigherRoleThan($token, $ip, $email, $user_id)
    {
        return $this->execute(__FUNCTION__, get_defined_vars());
    }

    /**
     * @param $token
     * @param $ip
     * @param $email
     * @param $user_id
     * @return BaseResponseInterface
     */
    public function getRoles($token, $ip, $email, $user_id)
    {
        return $this->execute(__FUNCTION__, get_defined_vars());
    }

    /**
     * @param $token
     * @param $ip
     * @param $email
     * @param $role_id
     * @return BaseResponseInterface
     */
    public function getPermissions($token, $ip, $email, $role_id)
    {
        return $this->execute(__FUNCTION__, get_defined_vars());
    }

    /**
     * @param $token
     * @param $ip
     * @param $email
     * @param $name
     * @param $description
     * @param array $permissions
     * @return BaseResponseInterface
     */
    public function storeRole($token, $ip, $email, $name, $description, array $permissions = [])
    {
        return $this->execute(__FUNCTION__, get_defined_vars());
    }

    /**
     * @param $token
     * @param $ip
     * @param $email
     * @param $role_id
     * @param $name
     * @param $description
     * @param array $permissions
     * @return BaseResponseInterface
     */
    public function updateRole($token, $ip, $email, $role_id, $name, $description, array $permissions = [])
    {
        return $this->execute(__FUNCTION__, get_defined_vars());
    }

    /**
     * @param $token
     * @param $ip
     * @param $email
     * @param $role_id
     * @return BaseResponseInterface
     */
    public function deleteRole($token, $ip, $email, $role_id)
    {
        return $this->execute(__FUNCTION__, get_defined_vars());
    }

    /**
     * @param $token
     * @param $ip
     * @param $email
     * @param $user_id
     * @return BaseResponseInterface
     */
    public function getRolesForUser($token, $ip, $email, $user_id)
    {
        return $this->execute(__FUNCTION__, get_defined_vars());
    }

    /**
     * @param $token
     * @param $ip
     * @param $email
     * @param $user_id
     * @return BaseResponseInterface
     */
    public function getPermissionsForUser($token, $ip, $email, $user_id)
    {
        return $this->execute(__FUNCTION__, get_defined_vars());
    }

    /**
     * @param $token
     * @param $ip
     * @param $email
     * @return BaseResponseInterface
     */
    public function getUsersRolesAndPermissions($token, $ip, $email)
    {
        return $this->execute(__FUNCTION__, get_defined_vars());
    }

    /**
     * @param $token
     * @param $ip
     * @param $email
     * @param $query
     * @return BaseResponseInterface
     */
    public function searchForUser($token, $ip, $email, $query)
    {
        return $this->execute(__FUNCTION__, get_defined_vars());
    }

    /**
     * @param $token
     * @param $ip
     * @param $email
     * @param $user_id
     * @param array $roles
     * @param array $permissions
     * @return BaseResponseInterface
     */
    public function assignOrRevokeRoles($token, $ip, $email, $user_id, array $roles = [], array $permissions = [])
    {
        return $this->execute(__FUNCTION__, get_defined_vars());
    }

    /** ################################################ Banlists & Whitelists Actions ################################################ */

    /**
     * @param $token
     * @param $ip
     * @param $email
     * @param $match_string
     * @param $page
     * @return BaseResponseInterface
     */
    public function getEmailBanlist($token, $ip, $email, $match_string, $page)
    {
        return $this->execute(__FUNCTION__, get_defined_vars());
    }

    /**
     * @param $token
     * @param $ip
     * @param $email
     * @param $emails
     * @return BaseResponseInterface
     */
    public function addEmailBanlist($token, $ip, $email, $emails)
    {
        return $this->execute(__FUNCTION__, get_defined_vars());
    }

    /**
     * @param $token
     * @param $ip
     * @param $email
     * @param $emails
     * @return BaseResponseInterface
     */
    public function editEmailBanlist($token, $ip, $email, $emails)
    {
        return $this->execute(__FUNCTION__, get_defined_vars());
    }

    /**
     * @param $token
     * @param $ip
     * @param $email
     * @param $emails
     * @return BaseResponseInterface
     */
    public function removeEmailBanlist($token, $ip, $email, $emails)
    {
        return $this->execute(__FUNCTION__, get_defined_vars());
    }

    /**
     * @param $token
     * @param $ip
     * @param $email
     * @param $match_string
     * @param $page
     * @return BaseResponseInterface
     */
    public function getIpBanlist($token, $ip, $email, $match_string, $page)
    {
        return $this->execute(__FUNCTION__, get_defined_vars());
    }

    /**
     * @param $token
     * @param $ip
     * @param $email
     * @param $ip_adresses
     * @return BaseResponseInterface
     */
    public function addIpBanlist($token, $ip, $email, $ip_adresses)
    {
        return $this->execute(__FUNCTION__, get_defined_vars());
    }

    /**
     * @param $token
     * @param $ip
     * @param $email
     * @param $ip_adresses
     * @return BaseResponseInterface
     */
    public function editIpBanlist($token, $ip, $email, $ip_adresses)
    {
        return $this->execute(__FUNCTION__, get_defined_vars());
    }

    /**
     * @param $token
     * @param $ip
     * @param $email
     * @param $ip_adresses
     * @return BaseResponseInterface
     */
    public function removeIpBanlist($token, $ip, $email, $ip_adresses)
    {
        return $this->execute(__FUNCTION__, get_defined_vars());
    }

    /**
     * @param $token
     * @param $ip
     * @param $email
     * @param $match_string
     * @param $page
     * @return BaseResponseInterface
     */
    public function getEmailWhitelist($token, $ip, $email, $match_string, $page)
    {
        return $this->execute(__FUNCTION__, get_defined_vars());
    }

    /**
     * @param $token
     * @param $ip
     * @param $email
     * @param $emails
     * @return BaseResponseInterface
     */
    public function addEmailWhitelist($token, $ip, $email, $emails)
    {
        return $this->execute(__FUNCTION__, get_defined_vars());
    }

    /**
     * @param $token
     * @param $ip
     * @param $email
     * @param $emails
     * @return BaseResponseInterface
     */
    public function editEmailWhitelist($token, $ip, $email, $emails)
    {
        return $this->execute(__FUNCTION__, get_defined_vars());
    }

    /**
     * @param $token
     * @param $ip
     * @param $email
     * @param $emails
     * @return BaseResponseInterface
     */
    public function removeEmailWhitelist($token, $ip, $email, $emails)
    {
        return $this->execute(__FUNCTION__, get_defined_vars());
    }

    /**
     * @param $token
     * @param $ip
     * @param $email
     * @param $match_string
     * @param $page
     * @return BaseResponseInterface
     */
    public function getIpWhitelist($token, $ip, $email, $match_string, $page)
    {
        return $this->execute(__FUNCTION__, get_defined_vars());
    }

    /**
     * @param $token
     * @param $ip
     * @param $email
     * @param $ip_adresses
     * @return BaseResponseInterface
     */
    public function addIpWhitelist($token, $ip, $email, $ip_adresses)
    {
        return $this->execute(__FUNCTION__, get_defined_vars());
    }

    /**
     * @param $token
     * @param $ip
     * @param $email
     * @param $ip_adresses
     * @return BaseResponseInterface
     */
    public function editIpWhitelist($token, $ip, $email, $ip_adresses)
    {
        return $this->execute(__FUNCTION__, get_defined_vars());
    }

    /**
     * @param $token
     * @param $ip
     * @param $email
     * @param $ip_adresses
     * @return BaseResponseInterface
     */
    public function removeIpWhitelist($token, $ip, $email, $ip_adresses)
    {
        return $this->execute(__FUNCTION__, get_defined_vars());
    }

    /** ################################################ Admin Actions ################################################ */

    /**
     * Approve the specified user.
     *
     * @param $token
     * @param $email
     * @param $ip
     * @param $userId
     * @param bool $via_twofa
     * @return BaseResponseInterface
     */
    public function approveUser($token, $email, $ip, $userId, bool $via_twofa = false)
    {
        return $this->execute(__FUNCTION__, get_defined_vars());
    }

    /**
     * Disapprove the specified user.
     *
     * @param $token
     * @param $email
     * @param $ip
     * @param $userId
     * @return BaseResponseInterface
     */
    public function disapproveUser($token, $email, $ip, $userId)
    {
        return $this->execute(__FUNCTION__, get_defined_vars());
    }
}