<?php

namespace Intergo\AuthClient\Rules;

use Illuminate\Contracts\Validation\Rule;

class PasswordRule implements Rule
{
    const MIN_LENGTH = 15;
    const MAX_LENGTH = 32;

    const ERROR_MSG_INVALID_PASSWORD = 'Password needs to have at least one uppercase, one lowercase, one special character and one digit.';
    const ERROR_MSG_PREVIOUS_PASSWORD_USAGE = 'The new password matches an older password you were using. Please set a password that you have not used before';
    const ERROR_MSG_CURRENT_PASSWORD_USAGE = 'New password can not be same as old password.';

    private bool $minLengthPasses = true;
    private bool $maxLengthPasses = true;
    /**
     * Determine if the validation rule passes.
     *
     * @param string $attribute
     * @param mixed $value
     * @return bool
     */
    public function passes($attribute, $value)
    {
        //somehow it passes array and this is related with task https://app.asana.com/0/1202041647164570/1202140521843719/f
        if (!is_string($value)) {
            return false;
        }

        if (strlen($value) < self::MIN_LENGTH) {
            $this->minLengthPasses = false;
            return false;
        }

        if (strlen($value) > self::MAX_LENGTH) {
            $this->maxLengthPasses = false;
            return false;
        }

        /**
         * Password requires at least:
         *  1 digit
         *  1 uppercase letter
         *  1 lowercase letter
         *  1 special character
         */
        return preg_match('/[0-9]/', $value)
            && preg_match('/[A-Z]/', $value)
            && preg_match('/[a-z]/', $value)
            && preg_match('/[^A-Za-z0-9]/', $value);
    }

    /**
     * Get the validation error message.
     *
     * @return string
     */
    public function message()
    {
        if (!$this->minLengthPasses) {
            return self::getMessageMinLength(':attribute');
        }

        if (!$this->maxLengthPasses) {
            return self::getMessageMaxLength(':attribute');
        }

        return self::ERROR_MSG_INVALID_PASSWORD;
    }

    /**
     * @param string $attribute
     * @return string
     */
    public static function getMessageMinLength($attribute = 'password')
    {
        return sprintf('The %s must be at least %d characters.', $attribute, self::MIN_LENGTH);
    }

    /**
     * @param string $attribute
     * @return string
     */
    public static function getMessageMaxLength($attribute = 'password')
    {
        return sprintf('The %s may not be greater than %d characters.', $attribute, self::MAX_LENGTH);
    }
}
